﻿
-- module setup
local me = { name = "tank"}
local mod = thismod
mod[me.name] = me


--[[
TankFinder.lua

It is important to determine which player is tanking the mob of interest, because that player's threat determines when we will pull aggro. As such it is used for the Aggro Gain bar and associated data.

The obvious method of determining a mob's tank (i.e. the player who has aggro) is to check its target. However, many mobs change their target to use an ability on a secondary target, so this doesn't always work well. A better idea is to check who the mob is autoattacking. Autoattacks will continue on the aggro target while the mob is casting spells on a secondary target.

This module keeps a list of all the autoattacks against targets that are on the threat meter. It uses this data as well as target data to work out who the tank is.

]]

--[[
me.myparsers = { ... }

6 events: either MISSES or HITS, and either SELF or PARTY or CREATURE. When a mob attacks a non-party member of your raid group, it goes under CREATURE_VS_CREATURE.

For hits, there are 4 possible: Crits or Hits, and physical or spell damage.
For misses, there are 4 possible: Miss, dodge, parry, block. We ignore immune and absorb as unlikely.
]]
me.myparsers =
{
	-- Hits vs me
	{"me", "COMBATHITCRITOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_HITS"}, -- %s crits you for %d.
	{"me", "COMBATHITOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_HITS"}, -- %s hits you for %d.
	{"me", "COMBATHITSCHOOLOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_HITS"}, -- %s hits you for %d %s damage.
	{"me", "COMBATHITCRITSCHOOLOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_HITS"}, -- %s crits you for %d %s damage.
	
	-- Misses vs me
	{"me", "MISSEDOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_MISSES"}, -- %s misses you.
	{"me", "VSBLOCKOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_MISSES"}, -- %s attacks. You block.
	{"me", "VSDODGEOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_MISSES"}, -- %s attacks. You dodge. 
	{"me", "VSPARRYOTHERSELF", "CHAT_MSG_COMBAT_CREATURE_VS_SELF_MISSES"}, -- %s attacks. You parry.
	
	-- Hits vs party
	{"other", "COMBATHITCRITOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_HITS"}, -- %s crits %s for %d.
	{"other", "COMBATHITOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_HITS"}, -- %s hits %s for %d.
	{"other", "COMBATHITCRITSCHOOLOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_HITS"}, -- %s crits %s for %d %s damage.
	{"other", "COMBATHITSCHOOLOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_HITS"}, -- %s hits %s for %d %s damage.
	
	-- Misses vs party
	{"other", "MISSEDOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_MISSES"}, -- %s misses %s.
	{"other", "VSBLOCKOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_MISSES"}, -- %s attacks. %s blocks.
	{"other", "VSDODGEOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_MISSES"}, -- %s attacks. %s dodges.
	{"other", "VSPARRYOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_PARTY_MISSES"}, -- %s attacks. %s parries.
	
	-- Hits vs raid
	{"other", "COMBATHITCRITOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_HITS"}, -- %s crits %s for %d.
	{"other", "COMBATHITOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_HITS"}, -- %s hits %s for %d.
	{"other", "COMBATHITCRITSCHOOLOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_HITS"}, -- %s crits %s for %d %s damage.
	{"other", "COMBATHITSCHOOLOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_HITS"}, -- %s hits %s for %d %s damage.
	
	-- Misses vs raid
	{"other", "MISSEDOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_MISSES"}, -- %s misses %s.
	{"other", "VSBLOCKOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_MISSES"}, -- %s attacks. %s blocks.
	{"other", "VSDODGEOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_MISSES"}, -- %s attacks. %s dodges.
	{"other", "VSPARRYOTHEROTHER", "CHAT_MSG_COMBAT_CREATURE_VS_CREATURE_MISSES"}, -- %s attacks. %s parries.
}	

-- This is called by Regex.lua when an event triggers one of <me.myparsers>
me.onparse = function(identifier, ...)
	
	if identifier == "me" then
		me.reportattack(select(1, ...), UnitName("player"))
		
	else
		me.reportattack(select(1, ...), select(2, ...))
	end
	
end

--[[
me.reportattack(mob, target)

Called when a mob's autoattack is detected. If the attack is notable, record it in <me.recentattacks>.

<mob>		string; name of the mob that autoattacked
<target>	string; name of the mob's target for the autoattack
]]
me.reportattack = function(mob, target)

	-- If there is a master target, only take attacks where he matches the mob name.
	if mod.target.mastertarget and (mod.target.mastertarget ~= mob) then
		return
	end
	
	-- only take targets who are recorded in the threat list
	if mod.table.raiddata[target] == nil then
		return
	end
	
	-- check mob entry exists
	if me.recentattacks[mob] == nil then
		me.recentattacks[mob] = { }
	end
	
	-- update target entry
	me.recentattacks[mob][target] = GetTime()
	
end

--[[
me.recentattacks = 
{
	mobname1 = <mobdata1>,
	mobname2 = <mobdata2>,
}

<mobdata> = 
{
	playername1 = GetTime(),
	playername2 = GetTime(),
}
]]
me.recentattacks = { }

-- Our polled functions
me.onupdates = 
{
	updaterecentattacks = 0.2
}

--[[
me.updaterecentattacks()

Clears old entries from the <me.recentattacks> table. This is a polled function - see <me.myonupdates>.
]]
me.updaterecentattacks = function()

	local oldtime = GetTime() - 4.0
	local mobname, mobdata, playername, value

	for mobname, mobdata in pairs(me.recentattacks) do
		for playername, value in pairs(mobdata) do
			if value < oldtime then
				mobdata[playername] = nil
			end
		end
	end

end

--[[
mod.tank.determinemaintank()

Figures out which player is tanking the current mob. If we can identify the mob, then make a list of all the players it has attacked recently. If there is only 1, he is the tank. If there are 0, use his current target. If there are multiple, use his current target as well i guess. 

This method is called by external modules, e.g. RaidTable.

Returns: <mtname>, <range>
<mtname>	string; the name of the player who has the mob's aggro, or <nil> if unknown
<range>	string; if <mtname> is non-nil, either "melee" or "range"; otherwise <nil>.
]]
me.determinemaintank = function()
	
	local name, unitid = me.determinecurrentmob()
	
	-- if we can't even name the target, then we can't find the MT
	if name == nil then
		return nil
	end
	
	-- check for a single target
	local singletarget = me.getsingletarget(name)
	
	if singletarget then
		return singletarget, me.getrangefromunit(unitid)
	end
	
	-- no single target - we need to know his current target now
	if unitid == nil then
		-- can't say
		return nil
	end
	
	-- to get here, his autoattack data was inconclusive, but we know his current target, so just use that. (it may be nil)
	return UnitName(unitid .. "target"), me.getrangefromunit(unitid)
	
end

--[[
me.getrangefromunit(unitid)

Determines, either precisely or by guessing, whether we are in melee range of <unitid> or not. Currently it just checks for 10 yards. Inside is melee, outside range. Lol.

Returns: "melee" or "range"
]]
me.getrangefromunit = function(unitid)
	
	if unitid then
		if CheckInteractDistance(unitid, 2) then
			return "melee"
		else
			return "range"
		end
	end
	
	-- if we can't determine the target, we are in real trouble. Guess based on class / spec
	if mod.my.class == "priest" then
		return "range"
		
	elseif mod.my.class == "mage" then
		return "range"
		
	elseif mod.my.class == "warlock" then
		return "range"
		
	elseif mod.my.class == "warrior" then
		return "melee"
		
	elseif mod.my.class == "rogue" then
		return "melee"
		
	elseif mod.my.class == "druid" then
		
		-- check for bear or cat form
		local form = mod.my.getstanceindex()
		
		if form == 1 or form == 3 then
			return "melee"
		else
			return "range"
		end
		
	elseif mod.my.class == "hunter" then
		return "range"
		
	elseif mod.my.class == "shaman" then
		_, _, count = GetTalentTabInfo(2)
		
		if count > 30 then
			return "melee"
		else
			return "range"
		end
		
	elseif mod.my.class == "paladin" then
		
		_, _, count = GetTalentTabInfo(3)
		
		if count > 30 then
			return "melee"
		else
			return "range"
		end
	end
	
end


--[[
me.getsingletarget(mobname)

Returns: string; the name of the only player in the mob's recent attack list, OR nil if there are 0 or many.
]]
me.getsingletarget = function(mobname)

	if me.recentattacks[mobname] == nil then
		return
	end
	
	local name
	
	for player, _ in pairs(me.recentattacks[mobname]) do
		
		-- check for multiple
		if name then
			return
		end
	
		-- only one so far
		name = player
	end
	
	-- might be single player or nil
	return name
	
end


--[[
me.determinecurrentmob()

Figures out the name and unitid of the current mob. Might be <nil> if we're not sure. This method call may scan your raid's targets to find the unitid.

This method is called by <me.determinemaintank>.

Logic: If there is a master target, use that mob. Otherwise check your target, targettarget, targettargettarget for a mob. Otherwise check for a unique player in the autoattack history. Otherwise forget it; just assume #1 threat is the MT.

Returns: <name>, <unitid>
<name>	string; name of the mob. <nil> if we can't tell.
<unitid>	string; unitid of the mob. <nil> if we can't tell.
]]
me.determinecurrentmob = function()
	
	local name
	
	-- 1) Master target takes priority
	if mod.target.mastertarget then
		name = mod.target.mastertarget
	
	-- 2) If there is no master target, check your current target chain. We can return directly here
	elseif UnitIsEnemy("player", "target") then
		return UnitName("target"), "target"
		
	elseif UnitIsEnemy("player", "targettarget") then
		return UnitName("targettarget"), "targettarget"

	elseif UnitIsEnemy("player", "targettargettarget") then
		return UnitName("targettargettarget"), "targettargettarget"
	
	-- 3) Finally check for a unique target in the autoattack history
	else
		
		local mobname, mobdata, playername, value, mobtarget
	
		for mobname, mobdata in pairs(me.recentattacks) do
			
			-- duplicate
			if name then
				return nil
			end
			
			-- record mob's name. Now check there is only 1 target.
			name = mobname
						
			for playername, value in pairs(mobdata) do
			
				-- duplicate
				if mobtarget then
					return nil
				else
					mobtarget = playername
				end
			end
		end
	end
	
	-- can't find the name
	if name == nil then
		return nil
	end
	
	-- found the name, search targets for it
	if GetNumRaidMembers() > 0 then
		
		for x = 1, 40 do
			if UnitName("raid" .. x .. "target") == name then
				return name, "raid" .. x .. "target"
			end
		end
		
	else
		
		if UnitName("target") == name then
			return name, "target"
		end
		
		if GetNumPartyMembers() > 0 then
			
			for x = 1, 4 do
				if UnitName("party" .. x .. "target") == name then
					return name, "party" .. x .. "target"
				end
			end
		end
	end
	
	-- to get here, can't find unitid
	return name, nil
	
end