﻿
local me = { name = "target"}
local mod = thismod
mod[me.name] = me

--[[
KTM_MasterTarget.lua

Manages the master target. This is an optional mob to filter our threat on. If it is enabled (<me.mastertarget> is non-nil), only threat towards that target is counted.
]]

--[[
me.mytraces = 
{
	default = "info",
}
]]

--[[
---------------------------------------------------------------------------------------------
					Slash Commands: Set / Clear the Master Target
---------------------------------------------------------------------------------------------
]]
me.myconsole = 
{
	mastertarget = "sendmastertarget",
}
	
--[[
This function is activated by "/mod mastertarget"
]]
me.sendmastertarget = function()
	
	-- check permission
	if mod.net.checkpermission() == nil then
		return
	end
	
	-- do we have a target selected?
	if UnitExists("target") then
		
		-- set target = current
		mod.net.sendmessage("target " .. UnitName("target"))
	
	else
		-- clear target
		mod.net.sendmessage("cleartarget")
	end
	
end

--[[
---------------------------------------------------------------------------------------------
					Handling Network Messages
---------------------------------------------------------------------------------------------
]]

me.playerversions = { } -- <playername> = <data> = {enabled, version, revision, time}
me.higherversions = 0
me.equalversions = 0

me.onload = function()
	
	me.mttrigger = mod.trigger.createnew("mtsuggest", 3.0, 5.0, me.mttriggerfunction)
	
end 

me.myonupdates = 
{
	updateinfo = 60.0,
}

me.updateinfo = function()
	
	-- send my info
	mod.net.sendmessage(string.format("info %s %s %s", mod.string.mylocale, mod.global.release, mod.global.revision))
	
	-- update other people's info. 3 min timeout
	local timeout = GetTime() - 180
	local count = 0
	
	me.higherversions = 0
	me.equalversions = 0
	
	for player, data in pairs(me.playerversions) do
		if data.isenabled == true and data.lasttime < timeout then
			data.isenabled = false
		end
		
		if data.isenabled then
			-- collect version info
			if data.version > mod.global.release then
				me.higherversions = me.higherversions + 1
			
			elseif data.version == mod.global.release then
				if data.revision > mod.global.revision then
					me.higherversions = me.higherversions + 1
				
				elseif data.revision == mod.global.revision then
					me.equalversions = me.equalversions + 1
				end
			end
		end
		
		count = count + 1
	end		
	
	if mod.trace.check("info", me, "info") then
		mod.trace.printf("info check: higher = %s, equal = %s, total = %s.", me.higherversions, me.equalversions, count)
	end
	
end

me.mynetmessages = { "target", "cleartarget", "suggestmt", "info"}

me.onnetmessage = function(author, command, data)
	
	if command == "target" then
		
		-- check the author has permission
		if mod.unit.isplayerofficer(author) == nil then
			return "permission"
		end
		
		-- check newmt makes sense
		if data and string.len(data) > 0 then
			me.setrequest(author, data)
		
		else
			return "invalid"
		end
		
	elseif command == "cleartarget" then
		
		-- check the author has permission
		if mod.unit.isplayerofficer(author) == nil then
			return "permission"
		end
		
		me.clearrequest(author)
	
	elseif command == "info" then
		
		local locale, version, revision = string.match(data, "([a-zA-Z]+) (%d%d?) (%d+)")
		
		-- check valid syntax
		if revision == nil then
			return "invalid"
		end
		
		-- ignore players running a different locale
		if locale ~= mod.string.mylocale then
			return "locale"
		end
		
		-- cast
		version = tonumber(version)
		revision = tonumber(revision)
		
		-- ignore if we already have their version
		if me.playerversions[author] == nil then
			me.playerversions[author] = { }
		end
		
		me.playerversions[author].isenabled = true
		me.playerversions[author].version = version
		me.playerversions[author].revision = revision
		me.playerversions[author].lasttime = GetTime()
			
	elseif command == "suggestmt" then
		
		-- parse locale and mob name
		local locale, mobname = string.match(data, "([a-zA-Z]+) (.+)")
		
		-- check valid syntax
		if mobname == nil then
			return "invalid"
		end
		
		-- ignore players running a different locale
		if locale ~= mod.string.mylocale then
			return "locale"
		end
		
		if mod.trace.check("info", me, "mtsuggest") then
			mod.trace.printf("received mt suggestion of '%s' from %s.", mobname, author)
		end
		
		me.mttrigger:notify(author, mobname)
				
	end
		
end

--[[
This is called when the Trigger <me.mttrigger> procs.
]]
me.mttriggerfunction = function(value)
	
	if mod.trace.check("info", me, "mtsuggest") then
		mod.trace.print("mt trigger running with value = " .. tostring(value))
	end
	
	me.automastertarget(value)
	
end

--[[
This is called when a local boss mod indicates that we should change the master target to <target>.
]]
me.localsetmt = function(target)
	
	-- we can immediately set it for ourself
	me.automastertarget(target)
	
	-- decide whether to send it to the raid group or not (don't if it is cooling down)
	if mod.trace.check("info", me, "suggestmt") then
		mod.trace.printf("Boss Mods have identified %s as a master target.", target)
	end
	
	if me.mttrigger:isoncooldown() then
		
		if mod.trace.check("info", me, "suggestmt") then
			mod.trace.printf("SuggestMT is on cooldown; no self broadcast will occur.")
		end
		
		return
	end
	
	-- now schedule a broadcast in some time interval, depending on how many people have higher versions of the mod
	local minvalue = 0.2 * math.min(5, me.higherversions)
	local maxvalue = 0.3 * min(10, me.higherversions + me.equalversions)
	
	local value = minvalue + math.random() * (maxvalue - minvalue)
	
	mod.schedule.schedule("suggestmt", value, me.scheduledbroadcast)
	
	if mod.trace.check("info", me, "suggestmt") then
		mod.trace.printf("MT Broadcast will occur in %s seconds (between %s and %s).", value, minvalue, maxvalue)
	end
	
end

me.scheduledbroadcast = function()
	
	if mod.trace.check("info", me, "suggestmt") then
		mod.trace.printf("MT Broadcast proccing!")
	end
	
	-- check whether it's on cooldown again (it may have been broadcast and confirmed while we waited)
	if me.mttrigger:isoncooldown() then
		
		if mod.trace.check("info", me, "suggestmt") then
			mod.trace.printf("MT Broadcast aborted due to cooldown. Beaten!")
		end
		
		return
	end
	
	if mod.trace.check("info", me, "suggestmt") then
		mod.trace.printf("Sending MT Suggestion.")
	end
	
	mod.net.sendmessage(string.format("suggestmt %s %s", mod.string.mylocale, me.mastertarget))
	
end

---------------------------------------------------------------------------------------------

me.mastertarget = nil
me.isworldboss = false
me.lastmtauthor = "" -- name of the player, or "" for noone yet.
me.lastmttime = 0
me.lastpollmessage = ""

--[[
mod.target.automastertarget(target)
Called when the mod itself sets the mastertarget.
<target> is the localised name of the mob.
]]
me.automastertarget = function(target)

	-- ignore if this is a dupe
	if me.mastertarget == target then
		return
	end

	me.mastertarget = target
	mod.raidtable.updateview()
	
	me.isworldboss = true
	me.lastmttime = GetTime()
	me.lastpollmessage = ""
	
	-- explain to user
	mod.printf(mod.string.get("print", "boss", "automt"), target)
	
end

--[[
mod.target.bossdeath()
Called when players in the raid group report that the master target has died.
]]
me.bossdeath = function()
	
	me.mastertarget = nil
	mod.raidtable.updateview()
	
	me.isworldboss = false
	me.lastmttime = GetTime()
	me.lastpollmessage = ""
	
end

me.lastonupdate = 0
me.onupdateinterval = 10.0

-- resend MT if we were the sender
me.onupdate = function()
		
	-- update every few seconds only
	if GetTime() < me.lastonupdate + me.onupdateinterval then
		return
	else
		me.lastonupdate = GetTime()
	end
	
	-- poll MT only if we set it last
	if me.lastmtauthor ~= UnitName("player") then
		return
	end
	
	-- check if you are still an assistant
	if mod.unit.isplayerofficer(UnitName("player")) == nil then
		me.lastmtsender = ""
		return
	end
	
	-- send poll or clear
	if me.mastertarget then
		-- this is stopped till we get a better solution
		-- mod.net.sendmessage("mtpoll " .. me.mastertarget)
	
	else
		mod.net.sendmessage("cleartarget")
	end
	
end

--[[
mod.target.pollrequest(author, target)
Called when we receive a network message of someone polling (periodically rebroadcasting) the master target.

<author>		string; name of the player who sent the message
<target>		string; name of a mob
]]
me.pollrequest = function(author, target)
	
	-- this is disabled for the moment
	if true then
		return
	end
	
	-- ignore polls that come quickly after a target change (could be sync error)
	if GetTime() < me.lastmttime + 2.0 then
		return
	end
	
	-- ignore if this message is the same as the last message, or matches our current MT 
	if (target == me.lastpollmessage) or (target == me.mastertarget) then
		return
	end
	
	-- set but warn user
	me.mastertarget = target
	mod.raidtable.updateview()
	
	mod.printf(mod.string.get("print", "network", "mtpollwarning"), target, author)
	
	me.lastmttime = GetTime()
	me.lastmtstring = target
	me.lastmtauthor = author
	
	-- we can't tell if the mob is a worldboss or not, so assume not
	
end

--[[
mod.target.clearrequest(author, target)
Called when we receive a network message of someone clearing the master target.

<author>		string; name of the player who sent the message
]]
me.clearrequest = function(author)
	
	-- ignore polls that come quickly after a target change (could be sync error)
	if GetTime() < me.lastmttime + 2.0 then
		return
	end
	
	-- announce if there is a change
	if me.mastertarget then
		
		me.mastertarget = nil
		mod.raidtable.updateview()
		
		mod.printf(mod.string.get("print", "network", "mtclear"), author)
	end
	
	-- update value of author
	me.lastmtauthor = author
	me.lastmttime = GetTime()
	me.lastpollmessage = ""
	
end

--[[
mod.target.setrequest(author, target)
Called when we receive a network message of someone setting the master target.
The problem is that if you have a different localisation to <author>, you will think his target is spelt differently to <target>! So we have to check for this, and override if necessary.

<author>		string; name of the player who sent the message
<target>		string; name of a mob
]]
me.setrequest = function(author, target)
		
	local errormessage = ""	
	
	-- 1) Find the author's UnitID
	local officerunit = mod.unit.findunitidfromname(author)
	local officertarget = UnitName(tostring(officerunit) .. "target")
	
	-- If the officer's target is just far enough away from us, we will be able to target him but his name will be "Unknown", which could stuff things up
	if officertarget == UNKNOWN then
		officertarget = nil
	end
	
	-- 2) Check for differences
	if officertarget == nil then
		errormessage = string.format(mod.string.get("print", "network", "newmttargetnil"), target, author)
	
	elseif officertarget ~= target then
		errormessage = string.format(mod.string.get("print", "network", "newmttargetmismatch"), author, target, officertarget)
		target = officertarget
	end
		
	-- 3) Check for worldboss target
	if UnitClassification((officerunit or "") .. "target") == "worldboss" then
		me.isworldboss = true
	else
		me.isworldboss = false
	end
	
	-- 4) Notify user if there is a change
	if target ~= me.mastertarget then
		me.mastertarget = target
		mod.raidtable.updateview()
		
		-- print out any warning if there was one
		if errormessage ~= "" then
			mod.print(errormessage)
		end
		
		-- print out the new target
		mod.printf(mod.string.get("print", "network", "newmt"), target, author)
	end
	
	-- 5) update network parameters and such
	me.lastmtauthor = author
	me.lastmttime = GetTime()
	me.lastpollmessage = target
	
end

--[[
mod.boss.targetismaster(target)
Checks whether <target> is the master target. The master target is usually just a name / string, but it may be something
more general in the future (e.g. tracking both bosses in the Twin Emps fight).
<target> is the name of the mob being queried.
Return: non-nil if <target> is a mastertarget.
]]
--! This variable is referenced by these modules: combat, 
me.targetismaster = function(target)
	
	if me.mastertarget == nil then
		return true
	end
	
	if target == me.mastertarget then
		return true
	end
	
	-- insert other checks here, later.
	
	return -- (nil)
	
end
