﻿
local me = { name = "combat"}
local mod = thismod
mod[me.name] = me

--[[
KTM_Combat.lua

The combat module receives parsed combat log events, adds a few set bonuses, and calls methods from My.lua to work out the actual threat values.
]]

-- These are the events we would like to be notified of
me.myevents = { "CHAT_MSG_COMBAT_FRIENDLY_DEATH", }

-- these are kept for debug purposes. We need two because next attack abilities are split into two.
--! This variable is referenced by these modules: alert, 
me.lastattack = nil
me.secondlastattack = nil

--[[ 
This is a record of attacks in the last second while out of combat. We keep this because when you
go into combat by initiating an attack, the +combat event can come after the actual attack. 
]]
--! This variable is referenced by these modules: my, 
me.recentattacks = { } 

me.onupdate = function()

	local timenow = GetTime()
	local key
	local value

	for key, value in pairs(me.recentattacks) do
		if value[1] < timenow - 1 then
			me.recentattacks[key] = nil
		end
	end

end
	

-- This is a method level temporary variable. Declared at file level because it is a list,
-- and we don't want to keep paging heap memory every time he is created.
--! This variable is referenced by these modules: boss, 
me.event = 
{
	["hits"] = 0,
	["damage"] = 0,
	["threat"] = 0,
	["name"] = 0,
}

-- this stuff should NOT be in the combat module.

--[[ 
Special onevent() method that will be called by Core.lua:onevent()
]]
me.onevent = function()
	
	if event == "CHAT_MSG_COMBAT_FRIENDLY_DEATH" then
		
		if arg1 == UNITDIESSELF then -- UNITDIESSELF = "You die."
			-- death is a threat wipe
			mod.table.resetraidthreat()
			return true
		end
		
	end
		
end

--[[
mod.combat.specialattack(abilityid, target, damage, iscrit, spellschool)
This handles any attack from a spell that has special threat properties, i.e. all the spells in mod.data.spells .
<abilityid> is the internal identifier for these abilities. i.e. Sunder Armor has <abilityid> = "sunder". This is locale independent.
<damage>, <iscrit>, and <spellschool> are optional. <spellschool> is localised, and will have the value either nil or "" or one of SPELL_SCHOOL1_CAP, SPELL_SCHOOL2_CAP, etc.
<iscrit> is only accepted if it has the boolean value true.
]]
--! This variable is referenced by these modules: combatparser, 
me.specialattack = function(abilityid, target, damage, iscrit, spellschool, isdot)

	-- 1) check the attack is directed at the master target. If not, ignore.
	if mod.target.targetismaster(target) == nil then
		return
	end
	
	-- 2) get the player's global threat modifiers (defensive stance, blessing of salvation, etc)
	local threatmodifier = mod.my.globalthreat.value
	
	-- Now, most attacks can be handled gracefully by the table. However, for abilities that modify your autoattack, we would prefer to decouple the ability from the autoattack, so we have to handle these cases individually.
	
	-- reset me.event
	me.event.hits = 1
	me.event.damage = damage
	
	-- 3) Handle Autoattack modifying abilities separately
	if abilityid == "whitedamage" then
			
		-- normal behaviour
		me.event.threat = damage * threatmodifier
		
		-- shaman spiritual weapons
		if mod.my.class == "shaman" then
			me.event.threat = me.event.threat * mod.my.mods.shaman.meleethreat
		end
	
	-- Special case: lacerate. 
	elseif abilityid == "lacerate" then
	
		if isdot then
			me.event.threat = damage * mod.my.ability("lacerate", "multiplier") * threatmodifier
		else
			me.event.threat = (damage * mod.my.ability("lacerate", "multiplier") + mod.my.ability("lacerate", "threat")) * threatmodifier
		end
	
	-- Special case: Heroic Strike
	elseif abilityid == "heroicstrike" then
		
		local preimpaledamage = damage
		if iscrit == true then
			preimpaledamage = damage / (1 + mod.my.mods.warrior.impale)
		end
		
		local addeddamage = mod.my.ability("heroicstrike", "nextattack")
		local myaveragedamage = me.averagemainhanddamage()
		local whitedamage = preimpaledamage * (myaveragedamage / (addeddamage + myaveragedamage))
		
		-- Now make a separate method call for the autoattack component
		me.specialattack("whitedamage", target, whitedamage, nil, nil)
		
		-- The above method will have overwritten some parts of me.event, so change them back
		me.event.damage = damage - whitedamage
		me.event.threat = (me.event.damage + mod.my.ability("heroicstrike", "threat")) * threatmodifier
	
	-- Special Case: Maul
	elseif abilityid == "maul" then
		
		-- same as heroic strike, but a bit different
		local presavagefurydamage = damage / (1 + mod.my.mods.druid.savagefury)		
		local addeddamage = mod.my.ability("maul", "nextattack")
		local myaveragedamage = me.averagemainhanddamage()
		local whitedamage = presavagefurydamage * (myaveragedamage / (addeddamage + myaveragedamage))
		
		-- Now make a separate method call for the autoattack component
		me.specialattack("whitedamage", target, whitedamage, nil, nil)
		
		-- The above method will have overwritten some parts of me.event, so change them back
		me.event.damage = damage - whitedamage
		me.event.threat = threatmodifier * (damage + mod.my.ability("maul", "threat") - whitedamage)
		
	-- Default Case: all other abilities
	else
		
		local multiplier = mod.my.ability(abilityid, "multiplier")
		
		-- 2) Check for multiplier
		if multiplier then
			me.event.threat = me.event.damage * multiplier
			
		else
			me.event.threat = me.event.damage + mod.my.ability(abilityid, "threat")
		end
		
		-- 3) Multiply by global modifiers
		me.event.threat = me.event.threat * threatmodifier
	
	end
	
	-- Paladin righteous fury (can affect holy shield)
	if mod.my.class == "paladin" then
		
		-- righteous fury
		if spellschool == SPELL_SCHOOL1_CAP then -- holy
			me.event.threat = me.event.threat * mod.my.mods.paladin.righteousfury
		end
		
	elseif mod.my.class == "warlock" then
		
		if mod.data.spellmatchesset("Warlock Destruction", abilityid, nil) == true then
			
			-- warlock Nemesis 8/8 (can affect searing pain)
			if mod.my.mods.warlock.nemesis == true then
				me.event.threat = me.event.threat * 0.8
			end
			
			me.event.threat = me.event.threat * mod.my.mods.warlock.destructionthreat
		
		elseif mod.data.spellmatchesset("Warlock Affliction", abilityid, nil) == true then
			me.event.threat = me.event.threat * mod.my.mods.warlock.afflictionthreat
		end
		
		-- plagueheart bonuses
		if mod.my.mods.warlock.plagueheart == true then
			
			-- 1) 25% less for crits
			if iscrit == true then
				me.event.threat = me.event.threat * 0.75
			
			-- 2) 25% less for some dots
			elseif mod.data.spellmatchesset("Plagueheart 6 Bonus", abilityid, nil) == true then
				me.event.threat = me.event.threat * 0.75
			end
		end
	
	elseif mod.my.class == "shaman" then
		
		-- 2 = fire, 3 = nature, 4 = frost
		if (spellschool == SPELL_SCHOOL2_CAP) or (spellschool == SPELL_SCHOOL3_CAP) or (spellschool == SPELL_SCHOOL4_CAP) then
			
			me.event.threat = me.event.threat * mod.my.mods.shaman.spelldamagethreat
		end

	elseif mod.my.class == "druid" then
		
		if spellschool and spellschool ~= "" then
			me.event.threat = me.event.threat * mod.my.mods.druid.subtlety
		end		
	end
		
	-- check for >= 0 threat
	if me.event.threat + mod.table.getraidthreat() < 0 then
		me.event.threat = - mod.table.getraidthreat()
	end
	
	-- relocalise.
	if abilityid == "whitedamage" then
		me.event.name = mod.string.get("threatsource", "whitedamage")
		
	else
		me.event.name = mod.string.get("spell", abilityid)
	end
	
	-- Add to data
	me.addattacktodata(me.event.name, me.event)
	me.addattacktodata(mod.string.get("threatsource", "total"), me.event)
	
end

-- to work out which part of a next attack ability was from white damage
-- used by nextattack abilities like maul and heroic strike.
me.averagemainhanddamage = function()

	local min, max = UnitDamage("player")
	return (min + max) / 2

end

--[[
me.normalattack(spellname, damage, target, iscrit, spellschool)
Handles a damage-causing ability with no special threat properties. We often have to make modifiers for gear or talents here. Note that melee autoattacks are sent to <me.specialsttack()> instead.
<spellname> is the name of the ability.
<spellid> is the internal name for "special" spells or abilities, i.e. those we have to look out for because there are specific set bonuses or talents that affect them.
<damage> is the damage done.
<target> is the name of the mob you hit.
<iscrit> is a boolean, whether the hit was a critical one. Triggers iff it is the boolean value true.
<spellschool> is a string, one of SPELL_SCHOOL1_CAP, SPELL_SCHOOL2_CAP, etc, or possibly nil or "".
]]
--! This variable is referenced by these modules: combatparser, 
me.normalattack = function(spellname, spellid, damage, isdot, target, iscrit, spellschool)
	
	-- check the attack is directed at the master target. If not, ignore.
	if mod.target.targetismaster(target) == nil then
		return
	end

	-- threatmodifier includes global things, like defensive stance, tranquil air totem, rogue passive modifier, etc.
	local threatmodifier = mod.my.globalthreat.value
	
	-- Special threat mod: priest silent resolve (non-shadow spells only)
	if (mod.my.class == "priest") and (spellschool ~= SPELL_SCHOOL5_CAP) then
		threatmodifier = threatmodifier * (1.0 +  mod.my.mods.priest.silentresolve)
	end 
	
	-- Special threat modifiers for mages
	if mod.my.class == "mage" then
		
		if (spellschool == SPELL_SCHOOL6_CAP) or (mod.string.mylocale == "frFR" and spellschool == "Arcane") then -- arcane
			threatmodifier = threatmodifier * (1.0 + mod.my.mods.mage.arcanethreat)
		
		elseif spellschool == SPELL_SCHOOL4_CAP then -- frost
			threatmodifier = threatmodifier * (1.0 + mod.my.mods.mage.frostthreat)
		
		elseif spellschool == SPELL_SCHOOL2_CAP then -- fire
			threatmodifier = threatmodifier * (1.0 + mod.my.mods.mage.firethreat)
		end
	end
	
	-- Default values for me.event:
	me.event.hits = 1
	me.event.damage = damage
	me.event.threat = damage * threatmodifier

	-- now get the name:
	if spellid == "dot" then
		
		me.event.hits = 0
		me.event.name = mod.string.get("threatsource", "dot")
		
	elseif spellid == "damageshield" then
		me.event.name = mod.string.get("threatsource", "damageshield")
		
	else
		me.event.name = mod.string.get("threatsource", "special")
		me.event.threat = damage * threatmodifier
	end
	
	-- Now apply class-specific filters
	
	-- warlock
	if mod.my.class == "warlock" then
			
		if mod.data.spellmatchesset("Warlock Destruction", spellid, spellname) == true then
			
			-- warlock Nemesis 8/8 (can affect searing pain)
			if mod.my.mods.warlock.nemesis == true then
				me.event.threat = me.event.threat * 0.8
			end
			
			me.event.threat = me.event.threat * mod.my.mods.warlock.destructionthreat
		
		elseif mod.data.spellmatchesset("Warlock Affliction", spellid, spellname) == true then
			me.event.threat = me.event.threat * mod.my.mods.warlock.afflictionthreat
		end
		
		-- plagueheart bonuses
		if mod.my.mods.warlock.plagueheart == true then
			
			-- 1) 25% less for crits
			if iscrit == true then
				me.event.threat = me.event.threat * 0.75
			
			-- 2) 25% less for some dots
			elseif mod.data.spellmatchesset("Plagueheart 6 Bonus", spellid, spellname) == true then
				me.event.threat = me.event.threat * 0.75
			end
		end
	
	-- Priest
	elseif mod.my.class == "priest" then
			
		-- shadow affinity
		if spellschool == SPELL_SCHOOL5_CAP then
			me.event.threat = me.event.threat * mod.my.mods.priest.shadowaffinity
		end
		
		-- holy nova: no threat
		if spellid == "holynova" then
			me.event.threat = 0
		end
		
	-- Mage
	elseif mod.my.class == "mage" then
			
		-- netherwind
		if mod.my.mods.mage.netherwind == true then
			
			if (spellid == "frostbolt") or (spellid == "scorch") or (spellid == "fireball") then
				-- note that this won't trigger off the dot part of fireball, because then spellid will be "dot"
				me.event.threat = math.max(0, (me.event.threat - 100))
				
			elseif spellid == "arcanemissiles" then
				me.event.threat = math.max(0, (me.event.threat - 20))
			end
		end
		
		-- frostfire 8 piece proc
		if mod.my.states.notthere.value == true then
			me.event.threat = 0
			mod.my.setstate("notthere", false)
		end
		
	-- Rogue
	elseif mod.my.class == "rogue" then
		
		-- bonescythe 6/8
		if mod.my.mods.rogue.bonescythe == true then
			if (spellid == "sinisterstrike") or (spellid == "backstab") or (spellid == "eviscerate") or (spellid == "hemorrhage") then
				me.event.threat = me.event.threat * 0.92
			end
		end
		
	-- Paladin
	elseif mod.my.class == "paladin" then
		
		-- righteous fury
		if spellschool == SPELL_SCHOOL1_CAP then -- holy
			me.event.threat = me.event.threat * mod.my.mods.paladin.righteousfury
		end
	
	-- Shaman
	elseif mod.my.class == "shaman" then
		
		-- spiritual weapons
		if spellid == "stormstrike" then
			me.event.threat = me.event.threat * mod.my.mods.shaman.meleethreat
		end
		
		-- elemental precision
		-- 2 = fire, 3 = nature, 4 = frost
		if (spellschool == SPELL_SCHOOL2_CAP) or (spellschool == SPELL_SCHOOL3_CAP) or (spellschool == SPELL_SCHOOL4_CAP) then
			
			me.event.threat = me.event.threat * mod.my.mods.shaman.spelldamagethreat
		end
	
	elseif mod.my.class == "druid" then
		
		if spellschool and spellschool ~= "" then
			me.event.threat = me.event.threat * mod.my.mods.druid.subtlety
		end	
	
	end

	-- special: blood siphon no threat vs Hakkar. This may or may not be correct.
	if spellid == "bloodsiphon" then
		me.event.threat = 0
	end

	-- now add me.event to individual and totals
	me.addattacktodata(me.event.name, me.event)
	me.addattacktodata(mod.string.get("threatsource", "total"), me.event)
	
end

--[[
me.possibleoverheal(spellname, spellid, amount, target)
Works out the threat from a heal.
<spellname> is the localised name of the spell
<spellid> is the mod's internal name for the spell, if it is special (i.e. affected by talents / sets bonuses), otherwise "".
<amount> is the healed amount only (no overheal)
<target> is the name of the target
Called when you heal someone. Deducts the overhealing from the total, then calls the Heal method
]]
--! This variable is referenced by these modules: combatparser, 
me.possibleoverheal = function(spellname, spellid, amount, target)
	
	-- we can check the target's health, which will be the health before the heal. Then we work out what the 
	-- heal did, and we can calculate overheal.
	
	local unit = mod.unit.findunitidfromname(target)
	
	if unit == nil then
		if mod.trace.check("info", me, "healtarget") then
			mod.trace.printf("Could not find a UnitID for the name %s.", target)
		end
		-- (and assume there was no overheal)
	else
		local hpvoid = UnitHealthMax(unit) - UnitHealth(unit)
		amount = math.min(amount, hpvoid)
	end
	
	amount = math.max(0, amount)
	
	me.registerheal(spellname, spellid, amount, target)
	
end

--[[ 
me.registerheal(spellname, spellid, amount, target)
Works out the threat from a heal.
<spellname> is the localised name of the spell
<spellid> is the mod's internal name for the spell, if it is special (i.e. affected by talents / sets bonuses), otherwise "".
<amount> is the healed amount only (no overheal)
<target> is the name of the target
]]
me.registerheal = function(spellname, spellid, amount, target)

	-- in general, don't count heals towards the master target
	if mod.target.mastertarget and mod.target.targetismaster(target) then
		return
	end
		
	-- also, don't count heals towards a hostile target
	if (target == UnitName("target")) and UnitIsEnemy("player", "target") then
		return
	end
	
	me.event.hits = 1
	me.event.damage = amount
	
	local threatmod = mod.my.globalthreat.value
	
	-- Special threat mod: priest silent resolve (spells only)
	if mod.my.class == "priest" then
		threatmod = threatmod * (1 + mod.my.mods.priest.silentresolve)
	end 
	
	me.event.threat = amount * threatmod * mod.data.threatconstants.healing
	
	-- class-based healing multipliers
	if mod.my.class == "paladin" then
		me.event.threat = me.event.threat * mod.my.mods.paladin.healing
		
		me.event.threat = me.event.threat * mod.my.mods.paladin.righteousfury
		
	elseif mod.my.class == "druid" then
		me.event.threat = me.event.threat * mod.my.mods.druid.subtlety
		
		if spellid == "tranquility" then
			me.event.threat = me.event.threat * mod.my.mods.druid.tranquilitythreat
		end
		
	elseif mod.my.class == "shaman" then
		me.event.threat = me.event.threat * mod.my.mods.shaman.healing
		
		me.event.threat = me.event.threat * mod.my.mods.shaman.spelldamagethreat
		
	end 
	
	-- Special: healing abilities which don't cause threat
	if spellid == "holynova" then
		me.event.threat = 0
	elseif spellid == "siphonlife" then
		me.event.threat = 0
	elseif spellid == "drainlife" then
		me.event.threat = 0
	elseif spellid == "deathcoil" then
		me.event.threat = 0
	end
	
	me.addattacktodata(mod.string.get("threatsource", "healing"), me.event)
	
	me.event.damage = 0
	me.addattacktodata(mod.string.get("threatsource", "total"), me.event)
	
end

--[[
me.powergain(amount, powertype)
Calculates the threat from gaining energy / mana / rage.
<amount> is the amount of power gained.
<powertype> is "Mana" or "Rage" or "Energy", but the localised versions.
<spellid> is the spell or effect that caused the power gain.
]]
--! This variable is referenced by these modules: combatparser, 
me.powergain = function(amount, powertype, spellid)

	me.event.damage = amount
	me.event.hits = 1
	
	-- 1) Prevent "overheal" for power gain
	local maxgain = UnitManaMax("player") - UnitMana("player")
	amount = math.min(maxgain, amount)
	
	if powertype == mod.string.get("power", "rage") then
		me.event.threat = amount * mod.data.threatconstants.ragegain
		
	elseif powertype == mod.string.get("power", "energy") then
		me.event.threat = amount * mod.data.threatconstants.energygain
		
	elseif powertype == mod.string.get("power", "mana") then
		me.event.threat = amount * mod.data.threatconstants.managain 
		
	else
		return
	end
	
	-- Special: abilities which don't cause threat
	if spellid == "lifetap" then
		me.event.threat = 0
	end
		
	me.addattacktodata(mod.string.get("threatsource", "powergain"), me.event)
	
	-- now mod it a bit to work into total better
	me.event.damage = 0
	me.event.hits = 0
	
	me.addattacktodata(mod.string.get("threatsource", "total"), me.event)
		
end

--[[
mod.combat.lognormalevent(name, hits, damage, threat)

Wrapper for <me.addattacktodata()>
]]
me.lognormalevent = function(name, hits, damage, threat)
	
	me.event.hits = hits
	me.event.damage = damage
	me.event.threat = threat
	
	me.addattacktodata(name, me.event)
	
	-- add to total too
	me.addattacktodata(mod.string.get("threatsource", "total"), me.event)
	
end

--[[
me.addattacktodata(name, data)
Once the threat from an attack has been worked, add it.
<name> is the category to add the threat to
<data> is me.event, i think...
Heap Memory will be created when you call this method when out of combat (mostly healing). 
From the size of the list (two numbers in it), will be 50 bytes tops each time.
]]
--! This variable is referenced by these modules: boss, 
me.addattacktodata = function(name, data)
	
	-- Ignore if charmed
	if mod.my.states.playercharmed.value == true then
		return
	end 
	
	-- misdirection
	if mod.misdirection.isactive and mod.misdirection.iscaster and data.threat > 0 then
		
		if name == mod.string.get("threatsource", "total") then
			mod.net.sendmessage("misdirection " .. mod.misdirection.target .. " " .. math.floor(data.threat))
			
			-- don't append this to our total
			return
		else
			-- change the name to misdirection (also, don't add to total)
			name = mod.string.get("spell", "misdirection")
		end
		
	end 		
	
	if name ~= mod.string.get("threatsource", "total") then
		me.secondlastattack = me.lastattack
		me.lastattack = data
		
		-- Add this to the recent attacks list, if we are not in combat
		if mod.my.states.incombat.value == false then
			table.insert(me.recentattacks, {GetTime(), data.threat})
		end
	end
	
	-- Add a new column to mod.table.mydata, if it does not exist already
	if mod.table.mydata[name] == nil then
		mod.table.mydata[name] = mod.table.newdatastruct()
	end
	
	mod.table.mydata[name].hits = mod.table.mydata[name].hits + data.hits
	mod.table.mydata[name].threat = mod.table.mydata[name].threat + data.threat
	mod.table.mydata[name].damage = mod.table.mydata[name].damage + data.damage 
	
end
