﻿
local me = { name = "boss"}
local mod = thismod
mod[me.name] = me

--[[
KTM_Bosses.lua

This module contains all the code for special boss encounters, determining who has aggro, etc.
]]

me.isspellreportingactive = false
me.istrackingspells = false
me.bosstarget = ""

me.onyxia = 
{
	lastcheck = 0,
	lasttargetself = 0,
}

--[[
---------------------------------------------------------------------------------------------
			Boss Module Network Messages: Spell Reporting
---------------------------------------------------------------------------------------------
]]

me.mynetmessages = { "spellstart", "spellstop", "spelleffect", "spellvalue", "event", }

--[[
mod.boss.onnetmessage(author, command, data)

Called by NetIn.lua when we receive an addon message matching one of the commands in <me.mynetmessages>.

<author>		string; name of the player who sent the message,
<command>	string; one of the values in <me.mynetmessages>,
<data>		string; remaining text of the addon message.

Returns: non-nil for error.
]]
me.onnetmessage = function(author, command, data)
	
	-- spellstart - starting spell reporting
	if command == "spellstart" then
		
		-- check the author has permission
		if mod.unit.isplayerofficer(author) == nil then
			return "permission"
		end
		
		me.isspellreportingactive = true
		
		if author == UnitName("player") then
			me.istrackingspells = true
		end
		
		-- only print out if you are a tracker
		if me.istrackingspells == true then
			mod.printf(mod.string.get("print", "network", "knockbackstart"), author)
		end
	
	-- spellstop - disabling spell reporting
	elseif command == "spellstop" then
		
		-- check the author has permission
		if mod.unit.isplayerofficer(author) == nil then
			return "permission"
		end
		
		me.isspellreportingactive = false
	
		if me.istrackingspells == true then
			mod.printf(mod.string.get("print", "network", "knockbackstop"), author)
		end
	
	-- spelleffect - someone reports being hit by a boss ability
	elseif command == "spelleffect" then
		
		if me.istrackingspells == false then
			return -- ignore, we don't actually check correctness.
		end
		
		--[[
		the format of data is "<spellname> <bossname> <effect> <data>"
		<spellname> and <bossname> are both enclosed in quote marks, the " character, since they may contain internal spaces,
		<effect> summarises what happened,
		<data> is optional.
		
		^			specifies that the match must start at the beginning of the string
		\"			a quote character, "
		(.+)		one or more characters, greedy. This is the <spellname> bit
		(.*)		zero or more characters, greedy. This is the <bossname> bit, could be empty
		(%l+)		one or more letters. This is the <effect> bit
		 ?			zero or one space. Because <data> is optional.
		]]
		local _, _, spell, mob, result, data = string.find(data, "^\"(.+)\" \"(.*)\" (%l+) ?(.*)")
		
		-- check the parse worked
		if spell == nil then
			return "invalid"
		end
		
		-- "miss" means the ability missed, so there was no effect
		if result == "miss" then
			mod.printf(mod.string.get("print", "boss", "reportmiss"), author, mob, spell)
		
		else
			local value1, value2
			
			-- (-?%d+) extracts a possibly negative integer
			_, _, value1, value2 = string.find(tostring(data), "(-?%d+) (-?%d+)")
						
			if (tostring(value1) == nil) or (tostring(value2) == nil) then
				return "invalid" -- data is wrong
			end
			
			-- "proc" - the ability hit, threat goes from <value1> to <value2>
			if result == "proc" then
				mod.printf(mod.string.get("print", "boss", "reportproc"), author, mob, spell, value1, value2)
			
			-- "tick" - the ability hit for the <value1>th time, will proc in <value2> more hits.
			elseif result == "tick" then
				mod.printf(mod.string.get("print", "boss", "reporttick"), author, mob, spell, value1, value2)
				
			else
				return "invalid" -- result is wrong
			end
		end
	
	-- spellvalue - someone changes the parameters of a boss ability
	elseif command == "spellvalue" then
		
		-- check the author has permission
		if mod.unit.isplayerofficer(author) == nil then
			return "permission"
		end	
		
		-- argh!! Memory creation! 64 bytes for this method! ... 
		local arglist = {}
		local x
		
		for x in string.gmatch(data, "[^ ]+") do
			table.insert(arglist, x)
		end
		
		-- parse
		local value, errormsg = me.checkspellvaluesyntax(arglist)
		
		-- did it work?
		if value then
			if arglist[2] == "default" then
				
				--[[ 
				suppose we receive the message "spellvalue timelapse default ticks 6". Then this would print out
					"Kenco sets the ticks parameter of the Time Lapse ability to 6."
				]]
				mod.printf(mod.string.get("print", "boss", "spellsetall"), author, "|cffffff00" .. arglist[3] .. "|r", mod.string.get("boss", "spell", arglist[1]), tostring(value), tostring(mod.boss.bossattacks[arglist[1]][arglist[2]][arglist[3]]))
			
			else				
				mod.printf(mod.string.get("print", "boss", "spellsetmob"), author, "|cffffff00" .. arglist[3] .. "|r", mod.string.get("boss", "name", arglist[2]), mod.string.get("boss", "spell", arglist[1]), tostring(value), tostring(mod.boss.bossattacks[arglist[1]][arglist[2]][arglist[3]]))
			end
			
			mod.boss.bossattacks[arglist[1]][arglist[2]][arglist[3]] = value
		
		else
			
			if mod.trace.check("warning", me, "spellvalue") then
				mod.trace.printf("The error message was '%s'.", errormsg)
			end
			
			return "invalid"
		end
	
	elseif command == "event" then
		
		if data == nil or me.bossevents[data] == nil then
			return "invalid"
		end
		
		me.reportevent(data, author)
	
	end
	
end

--[[
---------------------------------------------------------------------------------------------
					Slash Commands: Changing Boss Abilities
---------------------------------------------------------------------------------------------
]]

me.myconsole = 
{
	boss = 
	{
		report = "startspellreporting",
		endreport = "stopspellreporting",
		setspell = "setspellvalue",
	}
}

--[[
This is the function from "/mod boss report"
]]
me.startspellreporting = function()

	if mod.net.checkpermission() then
		mod.net.sendmessage("spellstart")
	end
	
end

--[[
This is the function from "/mod boss endreport"
]]
me.stopspellreporting = function()
	
	if mod.net.checkpermission() then	
		mod.net.sendmessage("spellstop")
	end
	
end

--[[
This is the function from "/mod boss setspell [...]"
]]
me.setspellvalue = function(text)
	
	-- load the arguments into <args>
	text = text or ""
	local args = {}
	
	for x in string.gmatch(text, "[^ ]+") do
		table.insert(args, x)
	end
	
	-- check syntax
	local value, errormessage = me.checkspellvaluesyntax(args)

	if errormessage then
		mod.print("|cffff8888Syntax: setspell <spellid> <bossid> <parameter> <value>")
		mod.print(errormessage)
		return
	end
	
	-- check officer permission
	if mod.net.checkpermission() then
		mod.net.sendmessage(string.format("spellvalue %s %s %s %s", unpack(args, 1, 4)))
	end
	
end

-- This code is so horrible i don't want to look at it.
-- syntax: <spellid> <bossid> <parameter> <value>
--[[
	if it succeeds, it will return just the value that is set.
	if it fails, it will return nil, then the error message.
]]
--! This variable is referenced by these modules: console, netin, 
me.checkspellvaluesyntax = function(allvalues)
			
	local x, spellid, bossid, parameter, value, key, message

	-- Check their first argument, <spellid>, is valid
	spellid = allvalues[1]
	if (spellid == nil) or (mod.boss.bossattacks[spellid] == nil) then
		
		message = "The argument |cffffff00" .. tostring(spellid) .. "|r does not match any boss spell id. Valid spellids are|cffffff00"
		
		for key, value in pairs(mod.boss.bossattacks) do
			message = message .. " " .. key
		end

		message = message .. "|r."
		return nil, message
	end
	
	-- Check their second argument, <bossid>, is valid
	local dataset = me.bossattacks[spellid]
	
	bossid = allvalues[2]
	if (bossid == nil) or (dataset[bossid] == nil) then
		
		message = "The argument |cffffff00" .. tostring(bossid) .. "|r does not match any boss that uses the spell |cffffff00" .. mod.string.get("boss", "spell", spellid) .. "|r. Valid bossids are|cffffff00"
		
		for key, value in pairs(dataset) do
			message = message .. " " .. key
		end

		message = message .. "|r."
		return nil, message
	end
	
	-- Check their third argument, <parameter>, is valid
	dataset = dataset[bossid]
	
	parameter = allvalues[3]
	if (parameter == nil) or (dataset[parameter] == nil) then
		
		message = "The argument |cffffff00" .. tostring(parameter) .. "|r does not match any parameter that can be set. Valid parameters are|cffffff00"
		
		for key, valud in pairs(dataset) do
			message = message .. " " .. key
		end

		message = message .. "|r."
		return nil, message
	end
	
	-- 4th parameter is value
	value = allvalues[4]
	
	-- multiplier / addition: need number
	if (parameter == "addition") or (parameter == "multiplier") then
		value = tonumber(value)
		
		if value == nil then
			
			message = "The argument |cffffff00" .. tostring(allvalues[4]) .. "|r is not a number."
			return nil, message
		end
		
	elseif parameter == "ticks" then
		value = tonumber(value)
		
		if (value == nil) or (math.floor(value) ~= value) or (value < 1) then
			
			message = "The argument |cffffff00" .. tostring(allvalues[4]) .. "|r is not a positive integer."
			return nil, message
	end
		
	elseif parameter == "effectonmiss" then
		
		if value == "true" then
			value = true
		
		elseif value == "false" then
			value = false
		
		else
			message = "The argument |cffffff00" .. tostring(allvalues[4]) .. "|r is not a boolean value."
			return nil, message
		end
		
	elseif parameter == "type" then
		
		if (value ~= "physical") or (value ~= "debuff") or (value ~= "spell") then
			
			message = "The argument |cffffff00" .. tostring(allvalues[4]) .. "|r is not one of |cffffff00 physical debuff spell|r."
			return nil, message
		end
	end
	
	-- it worked!
	return value
end



--[[
---------------------------------------------------------------------------------------------
					Events - Catching Monster Emotes and Yells
---------------------------------------------------------------------------------------------
]]

me.myevents = { "CHAT_MSG_MONSTER_EMOTE", "CHAT_MSG_MONSTER_YELL", }

me.onevent = function()
	
	-- disabled for KTM 20
	if true then
		return
	end
	
	if event == "CHAT_MSG_MONSTER_EMOTE" then
		
		-- Razorgore phase 2
		if string.find(arg1, mod.string.get("boss", "speech", "razorphase2")) then
			
			-- clear threat when phase 2 starts
			mod.table.resetraidthreat()
			
			-- set the master target to Razorgore, but only if a localised version of him exists.
			local bossname = mod.string.get("boss", "name", "razorgore")
			
			if mod.string.unlocalise("boss", "name", bossname) then
				mod.target.automastertarget(bossname)
			end
			
			return
		end
	
	elseif event == "CHAT_MSG_MONSTER_YELL" then
	
		-- Nef Phase 2
		if string.find(arg1, mod.string.get("boss", "speech", "nefphase2")) then
			
			-- reset threat in phase 2
			mod.table.resetraidthreat()
			
			-- boss name is given by the arg2
			mod.target.automastertarget(arg2)
			
			return
		end	
	
		-- ZG Tiger boss phase 2
		if string.find(arg1, mod.string.get("boss", "speech", "thekalphase2")) then
			
			-- reset threat in phase 2
			mod.table.resetraidthreat()
			
			-- boss name is given by the arg2
			mod.target.automastertarget(arg2)
			
			return
		end
		
		-- Rajaxx attacks
		if string.find(arg1, mod.string.get("boss", "speech", "rajaxxfinal")) then
			
			-- reset threat when he finally attacks
			mod.table.resetraidthreat()
			
			-- boss name is given by arg2
			mod.target.automastertarget(arg2)
			
			return
		end
		
		-- Azuregos Port
		if string.find(arg1, mod.string.get("boss", "speech", "azuregosport")) then
			
			-- 1) Find Azuregos
			local bossfound = false
			
			for x = 1, 40 do
				
				if UnitClassification("raid" .. x .. "target") == "worldboss" then
					if CheckInteractDistance("raid" .. x .. "target", 4) then
						mod.table.resetraidthreat()
					end
					
					bossfound = true
					break
				end	
			end
			
			-- couldn't find anyone targetting Azuregos. Better reset just to be sure.
			if bossfound == false then
				mod.table.resetraidthreat()
			end
			
			return
		end
		
		-- Attumen phase 3
		if string.find(arg1, mod.string.get("boss", "speech", "attumen3")) then
			mod.table.resetraidthreat()
			me.broadcastreset()
			mod.target.automastertarget(arg2)
		end
		
		-- Nightbane pull
		if string.find(arg1, mod.string.get("boss", "speech", "nightbane")) then
			mod.target.automastertarget(arg2)
		end
		
		-- Nightbane landing
		if string.find(arg1, mod.string.get("boss", "speech", "nightbane2")) or
			string.find(arg1, mod.string.get("boss", "speech", "nightbane2b")) then
				
			mod.table.resetraidthreat()
			me.broadcastreset()
		end
		
		-- Magtheridon phase 2
		if string.find(arg1, mod.string.get("boss", "speech", "magtheridon2")) then
			mod.table.resetraidthreat()
			me.broadcastreset()
			mod.target.automastertarget(arg2)
		end
		
		-- Hydross changes
		if string.find(arg1, mod.string.get("boss", "speech", "hydross2")) or
			string.find(arg1, mod.string.get("boss", "speech", "hydross3")) then
				
			mod.table.resetraidthreat()
			me.broadcastreset()
		end
		
	end
	
end

--[[
me.broadcastreset()

This is designed for compatability with older versions. Say there is a new boss mod to reset when a boss does a yell. If only half the raid has the new version of KTM, they will reset themselves but the rest of the raid won't, which would suck. 

Therefore we'll forge a "shazzrahgate" even whenever there is a boss reset. And they will just think Oh, Shazzrah has blinked, better reset!
]]
me.broadcastreset = function()
	
	-- notify the raid, if this event isn't on cooldown 
	if GetTime() < me.bossevents.shazzrahgate.lastoccurence + me.bossevents.shazzrahgate.cooldown then
		-- on cooldown. don't send
	else
		me.broadcastevent("shazzrahgate")
	end

end

--[[
---------------------------------------------------------------------------------------------
			Parser Events - Mob Abilities and Buffs
---------------------------------------------------------------------------------------------
]]

me.myparsers =
{
	-- this is for school spells or debuffs
	{"magicresist", "SPELLRESISTOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, -- "%s's %s was resisted."

	-- these two are for school spells only
	{"spellhit", "SPELLLOGSCHOOLOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, -- "%s's %s hits you for %d %s damage."
	{"spellhit", "SPELLLOGCRITSCHOOLOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, -- "%s's %s crits you for %d %s damage."

	-- spellboth is for abilities or school spells
	{"attackabsorb", "SPELLLOGABSORBOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, -- "You absorb %s's %s."
		
	-- ability hit / miss only works for physical spells.
	{"abilityhit", "SPELLLOGOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, 		-- "%s's %s hits you for %d."
	{"abilityhit", "SPELLLOGCRITOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"},	-- "%s's %s crits you for %d."
	{"abilityhit", "SPELLBLOCKEDOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"}, 	-- "%s's %s was blocked."
	{"abilitymiss", "SPELLDODGEDOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"},	-- "%s's %s was dodged."
	{"abilitymiss", "SPELLPARRIEDOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"},	-- "%s's %s was parried."
	{"abilitymiss", "SPELLMISSOTHERSELF", "CHAT_MSG_SPELL_CREATURE_VS_SELF_DAMAGE"},		-- "%s's %s misses you."
	
	{"debuffstart", "AURAADDEDSELFHARMFUL", "CHAT_MSG_SPELL_PERIODIC_SELF_DAMAGE"}, -- "You are afflicated by %s."
	{"debufftick", "AURAAPPLICATIONADDEDSELFHARMFUL", "CHAT_MSG_SPELL_PERIODIC_SELF_DAMAGE"}, -- "You are afflicted by %s (%d)."
	{"mobspellcast", "SPELLCASTGOOTHER", "CHAT_MSG_SPELL_CREATURE_VS_CREATURE_DAMAGE"},		-- "%s casts %s."
	{"mobbuffgain", "AURAADDEDOTHERHELPFUL", "CHAT_MSG_SPELL_PERIODIC_CREATURE_BUFFS"}, 		-- "%s gains %s."
	
	{"mobdeath", "UNITDIESOTHER", "CHAT_MSG_COMBAT_HOSTILE_DEATH"}, -- "%s dies."
}

--[[
me.onparse(identifier, [arg1, arg2, arg3, ...])

Called by Regex.lua when an event matching one of our parsers is received.
]]
me.onparse = function(identifier, ...)

	-- deactivated in favour of new boss mods
	if true then return end

	if identifier == "mobbuffgain" then

		-- Get Boss, Spell
		local boss, spell = select(1, ...), select(2, ...)
		
		-- noth blink
		if spell == mod.string.get("boss", "spell", "nothblink") then
			
			-- notify the raid, if this event isn't on cooldown 
			if GetTime() < me.bossevents.nothblink.lastoccurence + me.bossevents.nothblink.cooldown then
				-- on cooldown. don't send
			else
				me.broadcastevent("nothblink")
			end
		end
	
	elseif identifier == "mobspellcast" then
		
		-- Get Boss, Spell
		local boss, spell = select(1, ...), select(2, ...)
		
		-- twin teleport
		if spell == mod.string.get("boss", "spell", "twinteleport") then
			
			-- notify the raid, if this event isn't on cooldown 
			if GetTime() < me.bossevents.twinteleport.lastoccurence + me.bossevents.twinteleport.cooldown then
				-- on cooldown. don't send
			else
				me.broadcastevent("twinteleport")
			end
				
		-- gate of shazzrah
		elseif spell == mod.string.get("boss", "spell", "shazzrahgate") then

			-- notify the raid, if this event isn't on cooldown 
			if GetTime() < me.bossevents.shazzrahgate.lastoccurence + me.bossevents.shazzrahgate.cooldown then
				-- on cooldown. don't send
			else
				me.broadcastevent("shazzrahgate")
			end
		end
	
	elseif identifier == "mobdeath" then
		
		local mobname = select(1, ...)
		
		if (mobname == mod.target.mastertarget) and (mod.target.isworldboss == true) then
			
			-- notify the raid, if this event isn't on cooldown 
			if GetTime() < me.bossevents.bossdeath.lastoccurence + me.bossevents.bossdeath.cooldown then
				-- on cooldown. don't send
			else
				me.broadcastevent("bossdeath")
			end
		
		end
	
	else
		me.parsebossattack(identifier, ...)
	end
	
end

--[[
---------------------------------------------------------------------------------------------
					OnUpdates - Polled Checks
---------------------------------------------------------------------------------------------
]]

me.myonupdates = 
{
	updatetwinemperors = 0.5,
	updatetickcounters = 1.0,
	updateeventreports = 0.0,
	updatespellreporting = 0.0,
	checktriggers = 0.0,
	updateonyxia = 0.0,
}

-- if we are out of combat, reset the ticks on all boss abilities that have them
me.updatetickcounters = function()
	
	local key, key2, value
	
	if UnitAffectingCombat("player") == nil then
		
		for key, value in pairs(me.tickcounters) do
			for key2 in pairs(value) do
				value[key2] = 0
			end
		end
	end
		
end

-- clear out all old event reports - one report but no confirmations for 1.0 seconds.
me.updateeventreports = function()
	
	local key, value
	local timenow = GetTime()
	
	for key, value in pairs(me.bossevents) do
		if (value.reporter ~= "") and (timenow > value.reporttime + 1.0) then
			
			-- debug
			if mod.trace.check("warning", me, "event") then
				mod.trace.printf("The event %s has not been confirmed. It was reported by %s.", key, value.reporter)
			end
			
			-- remove the report
			value.reporter = ""
		end
	end
	
end

-- spellreporting: check for target changes
me.updatespellreporting = function()
	
	if (me.isspellreportingactive == true) and (me.istrackingspells == true) and mod.target.mastertarget then
		
		-- 1) find mt
		local x, newtarget
		
		for x = 1, 40 do
			name = UnitName("raid" .. x .. "target")
			if name == mod.target.mastertarget then
				
				-- get target^2
				newtarget = UnitName("raid" .. x .. "targettarget")
				
				if newtarget == nil then
					newtarget = "<none>"
				end
				
				break
			end
		end
		
		-- couldn't find the boss?
		if newtarget == nil then
			newtarget = "<unknown>"
		end
		
		-- report!
		if newtarget ~= me.bosstarget then
			
			-- find the threat of the old target
			local oldthreat = mod.table.raiddata[me.bosstarget]
			if oldthreat == nil then
				oldthreat = "?"
			end
			
			-- threat of the boss' new target
			local newthreat = mod.table.raiddata[me.bosstarget] 
			if newthreat == nil then
				newthreat = "?"
			end
			
			-- print
			mod.printf(mod.string.get("print", "boss", "bosstargetchange"), mod.target.mastertarget, me.bosstarget, oldthreat, newtarget, newthreat)
			
			-- update bosstarget
			me.bosstarget = newtarget
		end
	end
	
end

-- Onyxia - check for targettarget = self. Only start if Onyxia is the MT
me.updateonyxia = function()

	if mod.target.mastertarget == mod.string.get("boss", "name", "onyxia") then
			
		local timenow = GetTime()
		
		-- check at most once per second
		if timenow < me.onyxia.lastcheck + 1.0 then
			-- do nothing
			
		else
			me.onyxia.lastcheck = timenow
			
			-- scan raid for onyxia
			local unitname
			
			for x = 1, 40 do
				unitname = UnitName("raid" .. x .. "target")
				
				if unitname == mod.string.get("boss", "name", "onyxia") then -- found onyxia
					
					-- are you target target?
					if UnitIsUnit("player", "raid" .. x .. "targettarget") then
						me.onyxia.lasttargetself = timenow
					end
					
					break
				end
			end	
		end
	end
	
end

-- small script to check for twin teleport in 2.0. Period = 500ms.
me.updatetwinemperors = function()
	
	-- A) Master target set to one of the emps
	if (mod.target.mastertarget == mod.string.get("boss", "name", "twinempcaster")) or (mod.target.mastertarget == mod.string.get("boss", "name", "twinempmelee")) then
		
		-- B) Twin teleport not on the cooldown
		if GetTime() > me.bossevents.twinteleport.lastoccurence + me.bossevents.twinteleport.cooldown then
			
			local x, unitid
		
			for x = 1, 40 do
				unitid = "raid" .. x .. "target"
				
				if (UnitName(unitid) == mod.string.get("boss", "name", "twinempcaster")) or (UnitName(unitid) == mod.string.get("boss", "name", "twinempmelee")) then
					
					-- c) no target and health < 100
					if (UnitHealth(unitid) < 100) and (UnitName(unitid .. "target") == nil) then
						
						me.broadcastevent("twinteleport")
					end
					
					break
				end
			end
		end
	end
	
end

--[[
------------------------------------------------------------------------------------------------
							Boss Events - Sending and Receiving
------------------------------------------------------------------------------------------------

Boss Events are when a mob changes his threat against everyone after taking some action. Some players may be out of (combat log) range of the boss action, so we have nearby players report these special events to the rest of the raid.
There is a potential for abuse if someone in the raid group sends false boss event reports, which could make the raid group incorrectly reset their threat. To counter this we require two people to report the same event within a small time interval for it to be activated.
For each event in <me.bossevents>, we keep track of the person who first reported it, and the time they reported it. If the trace key "boss.fireevent" is enabled, the mod will print out who first reported the event and who confirmed it.
Insertion: players in the raid report events in the network channel, and <me.reportevent(...)> is called from the <netin> module.
Maintenance: no OnUpdate maintenance necessary.
]]

--! This variable is referenced by these modules: netin, 
me.bossevents = { }

--[[
me.addevent(eventid, cooldown)
Defines a new event in <me.bossevents>. This is just a helper method to create <me.bossevents>. Called at file load.
<eventid> is a localisation key in the "boss"-"spell" set.
<cooldown> is the minimum time between casts, extreme lower bound. Want it large enough to avoid spams. 1 sec would probably do.
]]
me.addevent = function(eventid, cooldown)
	
	me.bossevents[eventid] = 
	{
		["cooldown"] = cooldown,
		lastoccurence = 0, 	-- GetTime()
		reporter = "",
		reporttime = 0, 	-- GetTime()
		["eventid"] = eventid,
	}
	
end

-- define all possible events. These methods are called at file read time.
me.addevent("shazzrahgate", 5.0)
me.addevent("twinteleport", 20.0)
me.addevent("wrathofragnaros", 5.0)
me.addevent("nothblink", 5.0)
me.addevent("bossdeath", 5.0)
me.addevent("fourhorsemenmark", 5.0)

--[[
mod.boss.reportevent(eventid, player)
Called when someone in the raid reports a boss event.
<eventid> is the internal name of the event.
<player> is the name of the player who reported it.
]]
--! This variable is referenced by these modules: netin, 
me.reportevent = function(eventid, player)

	local eventdata = me.bossevents[eventid]
	local timenow = GetTime()
	
	-- ignore if the event is cooling down
	if timenow < eventdata.lastoccurence + eventdata.cooldown then
		return
	end
	
	-- has this been reported recently? If so it is now confirmed and we can run it.
	if (eventdata.reporter ~= "") and (eventdata.reporter ~= player) then
		me.fireevent(eventdata, player)
	
	-- always trust reports from yourself
	elseif player == UnitName("player") then
		me.fireevent(eventdata, player)
	
	-- some player reports a new event. wait for confirmation
	else
		eventdata.reporter = player
		eventdata.reporttime = timenow
	end

end

--[[
me.fireevent(eventdata, player)
Run when an event is confirmed. Does whatever the event does.
<eventdata> is an structure in <me.bossevents>
<player> is the name of the player who confirmed the event
]]
me.fireevent = function(eventdata, player)
	
	-- debug
	if mod.trace.check("info", me, "event") then
		mod.trace.printf("The event |cffffff00%s|r has occured. It was reported by %s and confirmed by %s.", eventdata.eventid, eventdata.reporter, player)
	end
	
	-- first reset the event's timers
	eventdata.lastoccurence = GetTime()
	eventdata.reporter = ""
	
	-- now actually do the event
	if eventdata.eventid == "shazzrahgate" then
		mod.table.resetraidthreat()
		
	elseif eventdata.eventid == "twinteleport" then
		mod.table.resetraidthreat()
		
		-- activate the proximity aggro detection trigger
		me.starttrigger("twinemps")
		
	elseif eventdata.eventid == "wrathofragnaros" then
		mod.table.resetraidthreat()
	
	elseif eventdata.eventid == "nothblink" then
		mod.table.resetraidthreat()
		
	elseif eventdata.eventid == "bossdeath" then
		mod.target.bossdeath()
		
	elseif eventdata.eventid == "fourhorsemenmark" then
		-- do a half wipe
		mod.table.raidthreatoffset = mod.table.raidthreatoffset - 0.5 * mod.table.getraidthreat()
	end
	
end

--[[
------------------------------------------------------------------------------------------------
				Triggers - Hard To Detect Events That Require Polling
------------------------------------------------------------------------------------------------

Some events have no easily defined actions such as a combat log event, and must be checked for periodically instead.
For example in the Twin Emperors encounter, after a teleport the closest person to each emperor is given a moderate amount of threat. The only way to see who received the threat is to see who the emperors target. However, after the teleport they become stunned and have no target, so we have to wait until the stun period has ended. So we make a trigger to periodically check them for new targets.
Each trigger has these properties:
<isactive>		boolean, whether the mod is checking the trigger
<startdelay>	time in seconds after the trigger is activated that the mod will start checking it.
<timeout>		time in seconds after the trigger has started that the mod should give up on it
<mystarttime>	when the most recent activation of the trigger occured.

The names and basic properties of triggers are defined in the variable <me.triggers>. This is a key-value list where the key is the internal name of the trigger, and the value is a structure with the variables described above.
To activate a trigger, call the <me.starttrigger(trigger)> function with the internal name of the trigger.
The code that will run each time a trigger is polled is contained in the variable <me.triggerfunctions>. This is a key-value list, where the key is the internal name and the value is the function that is run.
]]

me.triggers = 
{
	twinemps = 
	{
		isactive = false,
		startdelay = 0.5,
		timeout = 5.0,
		mystarttime = 0,
		data = 0,
	},
	autotarget = 
	{
		isactive = false,
		startdelay = 1.0,
		timeout = 300.0,
		mystarttime = 0,
		data = 0,
	}
}

--[[
me.starttrigger(trigger)
Activates a trigger. The mod will start periodically checking for it.
<trigger> is the mod's internal identifier of the trigger, and matches a key to me.triggers.
]]
--! This variable is referenced by these modules: console, 
me.starttrigger = function(trigger)
	
	-- debug check for trigger being defined. We should generalise this, since is happens in a few different places in the mod. i.e. "badidentifierargument"
	-- maybe also some kind of flood control to stop error messages spamming onupdate.
	
	if (trigger == nil) or (me.triggers[trigger] == nil) then
		
		-- report error
		if mod.trace.check("error", me, "trigger") then
			mod.trace.printf("There is no trigger |cffffff00%s|r.", trigger or "<nil>")
		end
		
		return
	end
	
	local triggerdata = me.triggers[trigger]
	triggerdata.isactive = true
	triggerdata.mystarttime = GetTime() + triggerdata.startdelay
	triggerdata.data = 0
	
	-- debug
	if mod.trace.check("info", me, "trigger") then
		mod.trace.printf("The |cffffff00%s|r trigger has been activated.", trigger)
	end
		
end

--[[
This variable gives the code that runs when an active trigger is checked. The keys are the internal names of triggers, that match keys in <me.triggers>.
The values are functions. The functions should return non-nil if the trigger is to be deactivated.
]]
me.triggerfunctions = 
{
	--[[ 
	We want to find out if we are being targetting by one of the emps. To do this we find the emperors by scanning the targets of everyone in the raid group. Then once we have an emps's target, we check whether that is us.
	It might occur that one emps' target is known but the other is not (not sure if this could happen). In this case the trigger should not end; we should keep checking until we know both targets.
	However, if one of the emps is targetting us, we can instantly give ourself threat and exit.
	The threat gained is set at 2000. This isn't confirmed, and is instead a bit of a guess.
	]]
	twinemps = function(triggerdata)
		
		local x, name, firstbossname, unitid
		local bosshits = 0
		local bosstargets = 0
		
		-- loop through everyone in the raid
		for x = 1, 40 do
			
			unitid = "raid" .. x .. "target"
			if UnitExists(unitid) and (UnitClassification(unitid) == "worldboss") then
				
				-- we've found an emperor. check if we've seen him before
				name = UnitName(unitid)
				
				if name ~= firstbossname then
					bosshits = bosshits + 1
					
					-- if this is the first boss we've seen, put his name up
					if bosshits == 1 then
						firstbossname = name
					end
					
					-- now find the player the boss is targetting
					unitid = unitid .. "target"
					
					if UnitExists(unitid) then
						bosstargets = bosstargets + 1
						
						if UnitIsUnit("player", unitid) then
							-- an emp is targetting us. give us a bit of threat.
							
							mod.combat.event.hits = 1
							mod.combat.event.threat = 2000
							mod.combat.event.damage = 0
							mod.combat.event.rage = 0
							
							mod.combat.addattacktodata(mod.string.get("threatsource", "threatwipe"), mod.combat.event)
							
							-- clear hits for total column
							mod.combat.event.hits = 0
							mod.combat.addattacktodata(mod.string.get("threatsource", "total"), mod.combat.event)
							
							-- if an emperor is targetting us, he will be the only one, and we have all the information we need, so we want the trigger to deactivate
							bosstargets = 2
							bosshits = 2
						end
					end
					
					-- if we have found 2 bosses now, there's no need to do more searching
					if bosshits == 2 then 
						break
					end
				end
			end
		end
		
		-- don't give up on the trigger until we have found both boss targets on one loop
		if bosstargets == 2 then
			return true
		end
	end,
	
	--[[
	Autotarget trigger runs when you run the command "/ktm boss autoatarget". When you next target a world boss, you will set the target and clear the meter.
	]]
	autotarget = function(triggerdata)
		
		if UnitExists("target") and (UnitClassification("target") == "worldboss") then
			
			-- found a target. now only activate if we've been targetting him for a while
			if triggerdata.data == 0 then
				triggerdata.data = GetTime()
				return
				
			else
				-- 500 ms minimum.
				if GetTime() < triggerdata.data + 0.5 then
					return
				end
			end
			
			-- found a target. Activate
			if mod.target.mastertarget == UnitName("target") then
				
				-- someone has already set the master target to this mob. In this case don't do anything.
				mod.printf(mod.string.get("print", "boss", "autotargetabort"), UnitName("target"))
				
			else
				mod.net.clearraidthreat()
				mod.net.sendmastertarget()
			end
			
			return true
		end	
		
	end
}

--[[
me.checktriggers()
Loops through all possible triggers, checking for active ones and running them if need be. This is called in the OnUpdate() method.
]]
me.checktriggers = function()

	local key, data
	local timenow = GetTime()
	
	for key, data in pairs(me.triggers) do
		
		-- ignore inactive triggers
		if data.isactive == true then
			
			-- stop the trigger if it has timed out
			if timenow > data.mystarttime + data.timeout then
				
				data.isactive = false
				
				-- debug
				if mod.trace.check("warning", me, "trigger") then
					mod.trace.printf("The trigger |cffffff00%s|r timed out.", key)
				end
				
			-- don't process the trigger if the start delay is not over
			elseif timenow < data.mystarttime then
				-- (do nothing)
				
			else
				-- ok, run a trigger check
				if me.triggerfunctions[key](data) then
					data.isactive = false
				end
			end
		end
	end
end

--[[
------------------------------------------------------------------------------------------------
			Parsing the Combat Log to Detect Boss Special Attacks and Spells
------------------------------------------------------------------------------------------------
]]

me.tickcounters = { }

--[[
me.parsebossattack(identifier, ...)

Called from <me.onparse()>.

Handles a combat log line that describes a boss's attack or spell against the player.
--> Stage one is to parse the message to find which formatting pattern the message matches, e.g. "magicresist" or
"spellhit" etc, or none (then just exit).
--> Stage two is to fill in <me.action>, whch descibes the important parts of the attack, using the formatting patter and the arguments captured by the pattern.
--> Then we check whether this attack is actually a threat modifying attack. For this to be the case, there would be a localisation string whose value is the attack name, and there would be an entry in <me.bossattacks> with the same key as the localisation key.
--> Next we identify the ability w.r.t. the mob. Does the ability only come from one mob, and if so is the mob who just attacked us the right one? This involves a check in the next level of <me.bossattacks>.
--> Now we know the specific attack performed against us, and we have to work out whether it triggered. If the ability does not trigger on a miss (e.g. Knock Away), we won't do anything. If the ability only triggers after a number of ticks (Time Lapse), it will only trigger if the correct number of ticks has passed.
--> If it triggers, we just change our threat by the right amount, then report the threat change in the <combat> and <table> modules.

<message> is the combat log line.
<event> is the chat message event <message> was received on.
Returns: nothing.
]]
me.parsebossattack = function(identifier, ...)

	-- interrupt: wrath of ragnaros
	if select(2, ...) == mod.string.get("boss", "spell", "wrathofragnaros") then

		-- notify the raid, if this event isn't on cooldown 
		if GetTime() < me.bossevents.wrathofragnaros.lastoccurence + me.bossevents.wrathofragnaros.cooldown then
			-- on cooldown. don't send
		else
			me.broadcastevent("wrathofragnaros")
		end
	end

	-- Set the mob and ability (always arg1 and arg2, except for debuffgain)
	if identifier == "debuffstart" or identifier == "debufftick" then
		me.resetaction("", select(1, ...))
	else	
		me.resetaction(select(1, ...), select(2, ...))
	end
	
	-- set the spell and hit types
	local description = me.attackdescription[identifier]

	if description == nil then
		mod.print("no description for " .. identifier)
		return
	end

	if description.ishit then me.action.ishit = true end
	if description.isspell then me.action.isspell = true end
	if description.isdebuff then me.action.isdebuff = true end
	if description.isphysical then me.action.isphysical = true end
	
	-- find a spellid, if it exists
	local spellid = mod.string.unlocalise("boss", "spell", me.action.ability)
	
	-- interrupt: four horsemen marks
	if spellid == "mark1" or spellid == "mark2" or spellid == "mark3" or spellid == "mark4" then
		-- notify the raid, if this event isn't on cooldown 
		if GetTime() < me.bossevents.fourhorsemenmark.lastoccurence + me.bossevents.fourhorsemenmark.cooldown then
			-- on cooldown. don't send
		else
			me.broadcastevent("fourhorsemenmark")
		end
		
		return
	end
	
	-- check whether this spellid has special behaviour associated with it
	if (spellid == nil) or (me.bossattacks[spellid] == nil) then
		return
	end
	
	-- Check for a mob match
	local spelldata
	
	local mobid = mod.string.unlocalise("boss", "name", me.action.mobname)
	
	if mobid and me.bossattacks[spellid][mobid] then
		-- there is a specific version of this spell for this particular mob
		spelldata = me.bossattacks[spellid][mobid]
	
	elseif me.bossattacks[spellid].default == nil then
		-- this mob does not match any of the mobs that have the ability
		return
		
	else
		spelldata = me.bossattacks[spellid].default
	end
	
	-- Now process the spell
	
	-- 1) Does the ability activate on a miss?
	if (me.action.ishit == false) and (spelldata.effectonmiss == false) then
		
		-- ability will not activate
		if mod.trace.check("info", me, "attack") then
			mod.trace.printf("%s's attack %s did not activate because it missed.", me.action.mobname, me.action.ability)
		end
		
		-- spell reporting
		if me.isspellreportingactive == true then
			me.reportspelleffect(me.action.ability, me.action.mobname, "miss")
		end
		
		return
	end
	
	-- 2) Check number of ticks
	local mytickdata
	
	if spelldata.ticks ~= 1 then
		
		-- create a list if none exists yet
		if me.tickcounters[me.action.ability] == nil then
			me.tickcounters[me.action.ability] = { }
		end
		
		if me.tickcounters[me.action.ability][me.action.mobname] == nil then
			me.tickcounters[me.action.ability][me.action.mobname] = 0
		end
		
		-- create an entry if none exists so far
		me.tickcounters[me.action.ability][me.action.mobname] = me.tickcounters[me.action.ability][me.action.mobname] + 1
		
		-- now, have we gone enough ticks?	
		if me.tickcounters[me.action.ability][me.action.mobname] < spelldata.ticks then
			
			-- not enough ticks
			if mod.trace.check("info", me, "attack") then
				mod.trace.printf("This is tick number %d of %s; it will activate in another %d ticks.", me.tickcounters[me.action.ability][me.action.mobname], me.action.ability, spelldata.ticks - me.tickcounters[me.action.ability][me.action.mobname])
			end
			
			-- spell reporting
			local value1 = me.tickcounters[me.action.ability][me.action.mobname]
			local value2 = spelldata.ticks - value1
			
			if me.isspellreportingactive then
				me.reportspelleffect(me.action.ability, me.action.mobname, "tick", value1, value2)
			end
			
			return
			
		else
			-- we just got enough ticks, so now reset to 0
			me.tickcounters[me.action.ability][me.action.mobname] = 0
			
		end
	end
	
	-- Interrupt: Fireball from Onyxia. When was your last target time?
	if spellid == "fireball" then
		-- only activate if you have been targetted within 4 seconds
		
		if GetTime() > me.onyxia.lasttargetself + 4.0 then
			-- don't activate. this was just stray aoe
			return
		end
	end	
	
	-- 2z) Don't do anything if there is meant to be no effect
	if (spelldata.multiplier == 1.0) and (spelldata.addition == 0.0) then
		return
	end
	
	-- 3) To get here, the ability is definitely activating
	if mod.trace.check("info", me, "attack") then
		mod.trace.printf("%s's %s activates, multiplying your threat by %s then adding %s.", me.action.mobname, me.action.ability, spelldata.multiplier, spelldata.addition)
	end
	
	-- compute new threat
	local newthreat = mod.table.getraidthreat() * spelldata.multiplier + spelldata.addition
	
	-- remember threat can't go below 0
	newthreat = math.max(0, newthreat)
	
	-- threat change is the (possibly negative) amount of threat that was added
	local threatchange = newthreat - mod.table.getraidthreat()
	
	-- spellreporting
	if me.isspellreportingactive then
		me.reportspelleffect(me.action.ability, me.action.mobname, "proc", math.floor(0.5 + mod.table.getraidthreat()), math.floor(0.5 + newthreat))
	end
	
	-- add to threat wipes section, but not to totals
	mod.combat.event.hits = 1
	mod.combat.event.damage = 0
	mod.combat.event.rage = 0
	mod.combat.event.threat = threatchange
	
	mod.combat.addattacktodata(mod.string.get("threatsource", "threatwipe"), mod.combat.event)
	
	-- now add it to your raid threat total (but not your personal threat total)
	mod.table.raidthreatoffset = mod.table.raidthreatoffset + threatchange
		
end

--[[
me.broadcastevent(eventmane)

Report a boss event to the raid group.

<eventname>	string; the internal id of the event.
]]
me.broadcastevent = function(eventname)
	
	mod.net.sendmessage("event " .. eventname)
	
end

--[[
me.reportspelleffect(spellname, bossname, result, value1, value2)

Reports the use of a special boss ability on you to the raid group. This would happen when someone has enabled spell reporting, by e.g. "/mod boss report". "Special Boss Ability" means abilities that are known by the mod to reduce threat.

<spellname>		string; the name of the NPC spell.
<bossname>		string; the name of the NPC that cast the spell.
<result>			string; the effect it had. "miss" if it was dodged / resisted / etc. "tick" if it hit us, but did not yet proc (due to needing more applications to have an affect). "proc" if it hit and reduced our threat.

<value1, value2>	integers; when <result> is "miss", they are nil. When <result> is "tick", <value1> is how many ticks we've had so far, and <value2> is how many more we need before a proc. When <result> is "proc", <value1> was our old threat, and <value2> is our new threat.
]]
me.reportspelleffect = function(spellname, bossname, result, value1, value2)
	
	if result == "miss" then
		mod.net.sendmessage(string.format("spelleffect \"%s\" \"%s\" %s", spellname, bossname, result))
		
	else
		mod.net.sendmessage(string.format("spelleffect \"%s\" \"%s\" %s %s %s", spellname, bossname, result, value1, value2))
	end
	
end

--[[
me.resetaction()
Sets the values of me.action to their defaults.
]]
me.resetaction = function(mobname, ability)

	me.action.mobname = mobname
	me.action.ability = ability
	me.action.ishit = false
	me.action.isphysical = false
	me.action.isdebuff = false
	me.action.isspell = false
	
end

me.action = 
{
	mobname = "",
	ability = "",
	ishit = false,
	isphysical = false,
	isdebuff = false,
	isspell = false,
}

-- Note that <ishit> defaults to false, so we only set it when it is true
me.attackdescription = 
{
	["magicresist"] = 
	{
		isspell = true,
		isdebuff = true,
	},
	["spellhit"] =
	{
		isspell = true,
		ishit = true,
	},
	["attackabsorb"] = 
	{
		isspell = true,
		isphysical = true,
		ishit = true,
	},
	["abilityhit"] = 
	{
		isphysical = true,
		ishit = true,
	},
	["abilitymiss"] = 
	{	
		isphysical = true,
	},
	["debuffstart"] = 
	{
		ishit = true,
		isdebuff = true,
	},
	["debufftick"] = 
	{
		ishit = true,
		isdebuff = true,
	},
}

--[[
Here is where you define all the boss' attacks that affect threat.
	The first key in me.bossattacks is the identifier of the spell. That is, mod.string.get("boss", "spell", <first key>) 
is the localised version.
	The second key deep specifies which mob the attack comes from. You can choose "default" to make it apply to all mobs,
or you can specify a mob id, which will override the "default" value. Mob id's recognised are all the keys in the 
"boss" -> "name" section of the localisation tree.
	So if you want to define a new attack name or boss name, you'll have to add a new key to the localisation tree in the
"boss" -> "spell" and "boss" -> "name" sections respectively.
	Inside each block, the follow parameters are defined:
	<multiplier> - a value that your threat is multiplier by. e.g. the standard Knock Away is -50% threat, so this would be a 
multiplier of 0.5. A complete threat wipe would be a multiplier of 0.
	<addition> - a flat value that is added to your threat. Can be positive or negative or 0.
	<effectonmiss> - a boolean value specifying whether the event triggers even when it is resisted or misses you.
	<ticks> - the number of times you must suffer the attack before your threat is changed. e.g. most knockbacks happen every
time so <ticks> = 1, but Time Lapse reduces your threat only after a certain number of applications.
	<type> - describes the attack. Can be "physical" or "spell" or "debuff". Not used by the mod at the moment: it will 
assume that if the name matches, it has found the right ability.
]]
--! This variable is referenced by these modules: net, netin, 
me.bossattacks = 
{
	knockaway = 
	{
		default = 
		{
			multiplier = 0.5,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "physical",
		},
		onyxia = 
		{
			multiplier = 0.75,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "physical",
		},
	},
	wingbuffet = 
	{
		default = 
		{
			multiplier = 0.5,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "physical",
		},
		onyxia = 
		{
			multiplier = 1.0,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "physical",
		},
	},
	timelapse = 
	{
		default = 
		{
			multiplier = 1.0,
			addition = 0,
			effectonmiss = false,
			ticks = 5,
			type = "debuff"
		}
	},
	hatefulstrike = 
	{
		default = 
		{
			multiplier = 1.0,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "physical"
		}
	},
	sandblast = 
	{
		default = 
		{
			multiplier = 0,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "spell"
		}
	},
	fireball = 
	{
		onyxia = 
		{
			multiplier = 0,
			addition = 0,
			effectonmiss = true,
			ticks = 1,
			type = "spell",
		},
		default = 
		{
			multiplier = 1.0,
			addition = 0,
			effectonmiss = false,
			ticks = 1,
			type = "spell",
		},
	},
}
