﻿
-- module setup
local me = { name = "event"}
local mod = thismod
mod[me.name] = me

--[[
Events.lua

19 Jun 2007

----------------------------------------------------------------------------
					Internal Events
----------------------------------------------------------------------------

Events.lua now exposes an Internal Event service. This allows one module to raise as event and another module to receive it. To create an event handler, add code like this:

		me.myinternalevents = 
		{
			my = { "redoglobalthreat" }
		}
				
		me.oninternalevent = function(source, message, ...)
			(...)
		end

The first part says "I want to handle the event 'redoglobalthreat' from the 'my' module. The second part is your callback function. <source> is the module that raised the event, e.g. 'my', <message> is the event, e.g. 'redoglobalthreat', and ... is any arguments generated by the event.

To raise an event, call the method <me.raiseinternalevent> in this module, e.g.

		mod.event.raiseinternalevent(me, "redoglobalthreat", 123, "bob")

The first argument is always <me>; a reference to the calling module. The second argument is the event. Additional arguments add information to the event.
		
		
----------------------------------------------------------------------------
					WoW Events
----------------------------------------------------------------------------

This module provides an <onevent> service to the rest of the addon.

First implement a list of strings "myevents" with the names of the events you want, e.g.

		me.myevents = { "CHAT_MSG_SPELL_SELF_BUFF", "CHAT_MSG_SYSTEM", }
	
Then implement a method "onevent", e.g.

		me.onevent = function()
			(...)
		end
		
No arguments will be passed to this method, so you'll have to refer to the global variables <event>, <arg1>, <arg2>, etc. Make sure not to edit them, since other addons might need them!
]]

--[[
------------------------------------------------------------------------------------------
			Data Structures to remember OnEvent subscriptions
------------------------------------------------------------------------------------------
]]

--[[ 
<me.events> records which module wants which events. e.g.

me.events = 
{
	["CHAT_MSG_SPELL_SELF_BUFF"] = {"combat", "regex"} -- combat and regex module register the event
	["CHAT_MSG_SYSTEM"] = {"netin"},
}
]]
me.events = { }

me.internalevents = { } --[[ 
me.internalevents = 
{
	<module name> = 
	{
		<event name> = 
		{
			{ list of modules that are handling the event }
		}
	}
}
]]


--[[
------------------------------------------------------------------------------------------
				Startup: Loading the Events Module
------------------------------------------------------------------------------------------

Loader.lua provides runs our startup functions. First <me.onload> is called, which creates our frame. Then <me.onmoduleload> is called with other modules as arguments, which will register all the necessary events on our frame. Then <me.onloadcomplete> is called, which creates the OnEvent script handler on our frame, which will start the event stream.
]]

--[[
me.onload() - special function called by Loader.lua

Creates the frame we will generate Events and OnUpdates with.
]]
me.onload = function()
	
	-- Create a frame to receive events and onupdates
	me.frame = CreateFrame("Frame", nil, mod.loader.frame)
	me.frame:Show()

end

--[[
me.onmoduleload() is a special function called by Core.lua when the addon starts up. It will be called once for each other module in the addon. We check whether they wish to receive events or onupdates, and implement them.
]]
me.onmoduleload = function(module)

	-- 1) Check for OnEvent subscription

	-- check for .onevent / .myevents mismatch
	if module.myevents and not module.onevent then
		if mod.trace.check("warning", me, "events") then
			mod.trace.printf("The module |cffffff00%s|r has a |cffffff00.myevents|r list but no |cffffff00.onevent|r function.", module.name)
		end
		
	elseif module.onevent and not module.myevents then
		if mod.trace.check("warning", me, "events") then
			mod.trace.printf("The module |cffffff00%s|r has a |cffffff00.onevent|r function but no |cffffff00.myevents|r list.", module.name)
		end
	
	-- normal modules: if they have both
	elseif module.myevents then
		
		local event, index, isnew, index2, knownmodule
		
		
		for index, event in pairs(module.myevents) do
			
			-- first level table is by the event
			if me.events[event] == nil then
				me.events[event] = { }
				me.frame:RegisterEvent(event)
			end
			
			-- prevent multiple occurrences of the same module-event definition
			isnew = true
			
			for index2, knownmodule in pairs(me.events[event]) do
				if knownmodule == module.name then
					isnew = false
					break
				end
			end
			
			if isnew then
				table.insert(me.events[event], module.name)
			end
		end
	end
	
	-- 1) Check for OnInternalEvent subscription

	-- check for .oninternalevent / .myinternalevents mismatch
	if module.myinternalevents and not module.oninternalevent then
		if mod.trace.check("warning", me, "events") then
			mod.trace.printf("The module |cffffff00%s|r has a |cffffff00.myinternalevents|r list but no |cffffff00.oninternalevent|r function.", module.name)
		end
		
	elseif module.oninternalevent and not module.myinternalevents then
		if mod.trace.check("warning", me, "events") then
			mod.trace.printf("The module |cffffff00%s|r has a |cffffff00.oninternalevent|r function but no |cffffff00.myinternalevents|r list.", module.name)
		end
	
	-- normal modules: if they have both
	elseif module.myinternalevents then
		
		local modulename, moduledata, index, event
		
		--[[ recall that the definition in <module> looks like
				me.myinternalevents = 
				{
					my = { "redoglobalthreat" }
				}
		]]
		for modulename, moduledata in pairs(module.myinternalevents) do
			
			if me.internalevents[modulename] == nil then
				me.internalevents[modulename] = { }
			end
			
			for index, event in pairs(moduledata) do
			
				if me.internalevents[modulename][event] == nil then
					me.internalevents[modulename][event] = { }
				end
			
				-- add this module
				table.insert(me.internalevents[modulename][event], module.name)
			end
			
		end
	end
	
end

--[[
me.onloadcomplete() is a special function called by Loader.lua after every module has loaded. This ensures that modules don't receive events until they have loaded.
]]
me.onloadcomplete = function()

	-- set handlers
	me.frame:SetScript("OnEvent", me.frameonevent)
		
	-- KLHBL compatability
	me.frame.klhblignore = true
	
end

--[[
------------------------------------------------------------------------------------------
				Runtime: Handling Events
------------------------------------------------------------------------------------------

Loader.lua provides runs our startup functions. First <me.onload> is called, which creates our frame. Then <me.onmoduleload> is called with other modules as arguments, which will register all the necessary events on our frame. Then <me.onloadcomplete> is called, which creates the OnEvent script handler on our frame, which will start the event stream.
]]

--[[
mod.event.lateaddevent(me, event)

Registers an event for a module. Call this if you need to receive a new event after the mod has already loaded. This method won't perform any correctness checking.

<module>		table; reference to the module that wants the event
<event>		string; the event, e.g. "CHAT_MSG_SYSTEM"
]]
me.lateaddevent = function(module, event)
	
	if me.events[event] == nil then
		me.events[event] = { }
		me.frame:RegisterEvent(event)
	end
	
	-- HELP LATER
	-- don't dup
	for x = 1, #me.events[event] do
		if me.events[event][x] == module.name then
			return
		end
	end

	table.insert(me.events[event], module.name)
	
end

--[[
me.frameonevent() is the OnEvent handler of <me.frame>.

We pass on the event to all modules that have registered for it.
]]
me.frameonevent = function()
	
	-- don't send events unless the mod is enabled and loaded
	if mod.loader.isloaded == false or mod.isenabled == false then
		return
	end
	
	-- get all the modules that have registered this event
	local subtable, index, key

	for index, key in pairs(me.events[event]) do
		
		subtable = mod[key]

		mod.diag.logmethodcall(key, "onevent")
	end
	
end

--[[
mod.event.raiseevent(module, eventname [, arg1, arg2, ...])

Generates an internal event.

-- <module> is a REFERENCE not a name.
]]
me.raiseevent = function(module, eventname, ...)
	
	if me.internalevents[module.name] and me.internalevents[module.name][eventname] then
		
		for index, modulename in pairs(me.internalevents[module.name][eventname]) do
		
			-- TODO: pcall wrapper on me!
			mod[modulename].oninternalevent(module.name, eventname, ...)
		end
	end
			
	
end