﻿
-- module setup
local me = { name = "sunder"}
local mod = thismod
mod[me.name] = me

--[[
KTM_Sunder.lua

This module handles the Sunder Armor spell.


------------------------------------------------
		World of Warcraft Implementation
------------------------------------------------

Sunder Armour is different from most spells because it is not accurately reported in the combat log. Only the status of the debuff on the mob is known, not who casts the spell or applications beyond the fifth. However Sunder Armor generates the same threat regardless of the number of debuffs on the target.

When we attempt to cast Sunder Armour, a UNIT_SPELLCAST_SENT event is raised with the name of the target. When the cast fires (no out of range or similar errors), a UNIT_SPELLCAST_SUCCEEDED event is raised.

When Sunder Armor is parried, dodged, misses, or the target is immune, we receive a CHAT_MSG_SPELL_SELF_DAMAGE event with format strings SPELLPARRIEDSELFOTHER, SPELLDODGEDSELFOTHER, SPELLMISSEDSELFOTHER, and IMMUNESPELLSELFOTHER respectively.


------------------------------------------------
		KTM Implementation
------------------------------------------------

When we receive UNIT_SPELLCAST_SENT event matching the spell we record the target (arg 4). When we receive a UNIT_SPELLCAST_SUCCESSFUL event matchint the spell we generate a standard attack in the <combat> module. If we detect a miss / parry / etc, we generate an attack in the <combat> module with -1 hits to deduct the threat.


------------------------------------------------
		Interaction With Other Modules
------------------------------------------------

Once a spell cast has been detected, we generate combat events just like the <combatparser> module would. These are sent to the <combat> module, which will use <my> and <data> modules for threat parameters.

The threat value for Sunder is kept in KTM_Data.lua in <me.spells.sunder>.

The following localisation key is required, with english example:

"spell" - "sunder" = "Sunder Armor"


----------------------------------------------------
				Debugging
----------------------------------------------------

To enable debugging of this module, uncomment these lines:

	me.mytraces = 
	{
		default = "info",
	}
	
]]

--[[
------------------------------------------------------------------
				Member Variables
------------------------------------------------------------------
]]

me.spellid = "sunder"	-- the internal id of this spell
me.spellname = ""			-- the localised name of this spell
me.lasttarget = ""		-- name of the most recent target of this spell

--[[
------------------------------------------------------------------
				Services from Loader.lua
------------------------------------------------------------------
]]

-- cache the name of this spell, mainly for code reuse purposes
me.onload = function()

	me.spellname = mod.string.get("spell", me.spellid)

end

--[[
------------------------------------------------------------------
				Services from Regex.lua
------------------------------------------------------------------
]]

me.myparsers = 
{
	{"parry", "SPELLPARRIEDSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"}, -- Your %s was parried by %s.
	{"miss", "SPELLMISSSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- Your %s missed %s.
	{"dodge", "SPELLDODGEDSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- Your %s was dodged by %s.
	{"immune", "IMMUNESPELLSELFOTHER", "CHAT_MSG_SPELL_SELF_DAMAGE"},	-- Your %s fails; %s is immmune.
}

me.onparse = function(identifier, spell, target)

	if spell == me.spellname then
		
		-- debug
		if mod.trace.check("info", me, "fail") then
			mod.trace.print(string.format("Sunder failed on %s, result was %s.", target, identifier))
		end
		
		me.retractcast()
	end

end

--[[
------------------------------------------------------------------
				Services from Events.lua
------------------------------------------------------------------
]]

me.myevents = { "UNIT_SPELLCAST_SUCCEEDED", "UNIT_SPELLCAST_SENT" }

me.onevent = function()

	-- check for spell match (arg2 for both events)
	if arg2 ~= me.spellname then
		return
	end

	if event == "UNIT_SPELLCAST_SENT" then
		me.lasttarget = arg4
		
		-- debug
		if mod.trace.check("info", me, "send") then
			mod.trace.print(string.format("sending sunder against %s.", arg4))
		end
	
	elseif event == "UNIT_SPELLCAST_SUCCEEDED" then
		
		-- debug
		if mod.trace.check("info", me, "send") then
			mod.trace.print(string.format("sunder has fired."))
		end
		
		me.submitcast()
	end

end	

--[[
------------------------------------------------------------------
				Methods
------------------------------------------------------------------
]]

--[[
me.submitcast()

Generates a spell event in the <combat> module.
]]
me.submitcast = function()

	mod.combat.specialattack(me.spellid, me.lasttarget, 0)

end

--[[
me.retractcast()

Generates a spell event in the <combat> module with a negative <hits> value, so the threat is subtracted. Unfortunately there's no facility in Combat.lua to retract an action, so we have to work out the threat manually.
]]
me.retractcast = function()

	-- 1) check for master target
	if mod.target.targetismaster(me.lasttarget) == nil then
		return
	end
	
	local threat = - mod.my.ability("sunder", "threat") * mod.my.globalthreat.value
	
	mod.combat.lognormalevent(me.spellname, -1, 0, threat)
	
end