﻿
local me = { name = "misdirection"}
local mod = thismod
mod[me.name] = me

--[[
Misdirection.lua

Handles the Hunter spell Misdirection.

----------------------------------------------------
		World of Warcraft Implementation
----------------------------------------------------

Misdirection is an instant cast spell. When it is successfully cast, both the caster and the target receive a buff called "Misdirection", reported as "CHAT_MSG_SPELL_PERIODIC_SELF_BUFFS".

While the buff is active, the next 3 attack by the hunter have their threat redirected to the target. The buff lasts for 30 seconds or 3 actions. The buff's fading is reported in "CHAT_MSG_SPELL_AURA_GONE_SELF".

----------------------------------------------------
			KTM Implementation
----------------------------------------------------

The first issue is to identify whether the player who receives the "Misdirection" buff is the caster or the target. To figure this out, we log "UNIT_SPELLCAST_SENT" events for Misdirection. When you receive the buff, <me.isactive> is set to true. If you had recently attempted to cast Misdirection, then <me.iscaster> is set to true; otherwise false.

Code has been added to Combat.lua to check whether misdirection is active and the player is the caster. If so, we send an addon message of the form "misdirection <target> <threat>". For example, "misdirection Kenco 900", if Kenco was the target of misdirection and the attack caused 900 threat. We got the value for <target> from the "UNIT_SPELLCAST_SENT" event.

The target receives the addon message in <me.onnetmessage>. They check if <target> matches their name. Then they will only increment their threat if <me.isactive> is true, or the buff has only faded recently.

<me.isactive> is set to false if the buff times out, or if the fade event is detected.

----------------------------------------------------
			Interaction with other modules
----------------------------------------------------

The method <mod.combat.addattacktodata> has been altered to support Misdirection (see above notes). The module requires services from the <event>, <regex> and <netin> modules. The target calls <mod.combat.lognormalevent> when their threat increases.

This module requires the localisation key ("spell", "misdirection") to be current.
]]

--[[
------------------------------------------------------------------------
				Member Variables
------------------------------------------------------------------------
]]

me.target = ""			-- who we targeted, if we cast it
me.isactive = false		-- whether the buff is active
me.starttime = 0.0		-- when we gained the buff
me.lasttrycast = 0.0	-- time we last tried to cast misdirection (SPELLCAST_SENT)
me.iscaster = false		-- whether we cast the misdirection (true) or received it (false)
me.fadetime = 0.0		-- when the buff last faded
	
-- constants
me.duration = 30.0		-- maximum buff duration

--[[
------------------------------------------------------------------------
			Services from Events.lua
------------------------------------------------------------------------
]]

me.myevents = { "UNIT_SPELLCAST_SENT" }

me.onevent = function()
	
	if arg2 == mod.string.get("spell", "misdirection") then
		me.target = arg4
		me.lasttrycast = GetTime()
	end
		
end

me.myonupdates = 
{	
	updatemisdirection = 1.0
}

me.updatemisdirection = function()
	
	-- safety timeout when duration expires
	if me.isactive and (GetTime() > me.starttime + me.duration) then
		me.isactive = false
	end
	
end

--[[
------------------------------------------------------------------------
			Services from Regex.lua
------------------------------------------------------------------------
]]

me.myparsers =
{
	{"buffstart", "AURAADDEDSELFHELPFUL", "CHAT_MSG_SPELL_PERIODIC_SELF_BUFFS"},	-- "You gain %s."
	{"buffend", "AURAREMOVEDSELF", "CHAT_MSG_SPELL_AURA_GONE_SELF"}, -- "%s fades from you."
}

me.onparse = function(identifier, ...)
		
	if identifier == "buffstart" then
				
		if select(1, ...) == mod.string.get("spell", "misdirection") then
			me.isactive = true
			me.starttime = GetTime()
			
			-- did we start it
			if me.starttime < me.lasttrycast + 2.0 then
				me.iscaster = true
			else
				me.iscaster = false
			end
			
		end
		
	elseif identifier == "buffend" then
		
		if select(1, ...) == mod.string.get("spell", "misdirection") then
			me.isactive = false
			me.lastfadetime = GetTime()
		end
		
	end
	
end

--[[
------------------------------------------------------------------------
			Services from NetIn.lua
------------------------------------------------------------------------
]]

me.mynetmessages = { "misdirection" }

me.onnetmessage = function(author, command, data)
	
	if command == "misdirection" then
		
		local _, _, player, value = string.find(data, "(.+) (%d+)")
		
		if player == nil then
			return "invalid"
		end
		
		if player == UnitName("player") then
			
			-- check whether misdirection is active on us
			if me.isactive or (GetTime() < me.fadetime + 2.0) then
				
				-- add threat			
				mod.combat.lognormalevent(mod.string.get("spell", "misdirection"), 1, 0, value)
							
			else
				return "invalid"
			end
		
		end
	end
	
end