﻿
local me = { name = "invisibility"}
local mod = thismod
mod[me.name] = me

--[[
Invisibility.lua

Handles the mage spell Invisibility. 

----------------------------------------------------
		World of Warcraft Implementation
----------------------------------------------------

Invisibility is an instant cast spell. When it is cast, you gain a buff called "Invisibility", BUT there is no entry in the combat log to signal this. You aren't yet invisible, only "fading...".

For 5 seconds the buff ticks down, and you lose 10% of your current threat each second. If you do or receive a hostile action, the buff fades (again no combat log entry) and you won't become invisible.

If the buff ticks down all they way, you do become invisible, and you gain a different buff, also called "Invisibility". This buff is reported in the combat log under "CHAT_MSG_SPELL_PERIODIC_SELF_BUFFS".


----------------------------------------------------
			KTM Implementation
----------------------------------------------------

We detect the spellcast from the UNIT_SPELLCAST_SUCCESSFUL event. Then <me.isactive> is true, and you lose <me.levelpertick> of your threat ever <me.tickinterval> seconds, to a maximum of <me.maxticks> ticks.

While ticking is ative, periodically we check whether the "Invisibility" buff has faded; if so <me.isactive> is set to false and the ticking stops.

If we detect the buff gain "Invisibility" from the combat log, we wipe threat, regardless of the value of <me.isactive>

----------------------------------------------------
			Interaction with other modules
----------------------------------------------------

The <invisibility> module uses services from the <regex> module and the <event> module. To change your threat it calls <mod.combat.lognormalevent> and <mod.table.resetraidthreat>

This module requires the localisation key ("spell", "invisibility") to be current.
]]

--[[
------------------------------------------------------------------------
				Member Variables
------------------------------------------------------------------------
]]

me.isactive = false		-- whether invisibility is currently ticking
me.casttime = 0			-- when SPELLCAST_SUCCESSFUL triggered
me.ticks = 0				-- number of ticks elapsed
	
-- these are constants
me.maxticks = 5			-- ticks until complete invisibility
me.levelpertick = 0.1 	-- fraction of threat reduced
me.tickinterval = 1.0	-- seconds between ticks

--[[
------------------------------------------------------------------------
			Services from Regex.lua
------------------------------------------------------------------------
]]

me.myparsers =
{
	{"buffstart", "AURAADDEDSELFHELPFUL", "CHAT_MSG_SPELL_PERIODIC_SELF_BUFFS"},	-- "You gain %s."
}

me.onparse = function(identifier, ...)
	
	if select(1, ...) == mod.string.get("spell", "invisibility") then
		me.isactive = false
		mod.table.resetraidthreat()
	end
	
end


--[[
------------------------------------------------------------------------
			Services from Events.lua
------------------------------------------------------------------------
]]

me.myevents = { "UNIT_SPELLCAST_SUCCEEDED" }

me.onevent = function()

    if arg2 == mod.string.get("spell", "invisibility") then
        me.isactive = true
        me.ticks = 0
        me.casttime = GetTime()
    end

end

me.myonupdates = 
{
	updateinvisibility = 0.1,
}

me.updateinvisibility = function()
	
	-- 1) if the buff is not active, nothing to do
	if me.isactive == false then
		return
	end
	
	-- 2) Check whether an action has caused the buff to fall off
    local ticksnow = math.floor((GetTime() - me.casttime) / me.tickinterval)
    
	if GetPlayerBuffName(mod.string.get("spell", "invisibility")) == nil and ticksnow > 0 then        
		me.isactive = false
		return
	end
	
	-- 3) Is it time for another tick?
	if ticksnow > me.ticks then
		me.ticks = me.ticks + 1
		
		if me.ticks > me.maxticks then
			me.isactive = false
			return
		
		else
			me.reducethreatfraction(me.levelpertick, mod.string.get("spell", "invisibility"))

		end
	end

end

--[[
me.reducethreatfraction(multiplier, ability)

<multiplier>	number; positive fraction, e.g. 0.2 for -20%.
<ability>		string; name of the ability used
]]
me.reducethreatfraction = function(multiplier, ability)
	
	local reduction = - mod.table.getraidthreat() * multiplier
	
	mod.combat.lognormalevent(ability, 1, 0, reduction)
	
end


